;+
; NAME:
;    mgs_basefile
;
; PURPOSE:
;    This is a generic object to handle I/O from data files. See
;    mgs_ncdffile for an application of this object.
;
; CATEGORY:
;    File I/O, Objects
;
; REQUIREMENTS:
;    Uses Arrex, ChkStru and StrRepl functions
;    Inherits from mgs_baseobject and uses mgs_container
;
; INHERITING:
;    Objects that want to use the basefile properties (i.e. 'real
;    world' file objects) should overwrite at least the following
;    methods:
;    * Init : set filetype accordingly
;    * IsOpen : perform a real test if file has been opened
;    * Read : nothing can be done in generic object
;    These methods are also likely to be overwritten:
;    * OpenAgain : if file type requires special open command
;    * Close : if file type requires special close command
;
; MODIFICATION HISTORY:
;    mgs, 05 Sep 2000: Version 0.1
;                      Limitations:
;                      - only read access supported
;                      - circular dimensions not supported (e.g. lon)
;                      - needs more rigorous testing
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2000 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################




;-----------------------------------------------------------------------------
; ValidTagName: 
;    This method replaces invalid characters in a string (e.g. a
; netCDF variable name) to allow building of a structure.

FUNCTION MGS_BaseFile::ValidTagName, arg

      t = strrepl(arg,'-','_')
      t = strrepl(t,'+','P')
      t = strrepl(t,'$','D')
      t = strrepl(t,'*','S')
      t = strrepl(t,'&','_')
      t = strrepl(t,' ','_')
      t = strrepl(t,'@','_')
      t = strrepl(t,'.','_')
      t = strrepl(t,',','_')
      t = strrepl(t,';','_')
      t = strrepl(t,'(','_')
      t = strrepl(t,')','_')
      t = strrepl(t,'[','_')
      t = strrepl(t,']','_')

      RETURN, t

END


; -----------------------------------------------------------------------------
; IsDimVar:
;   This function method returns 1 if the specified variable is a
; dimension variable, and 0 otherwise.

FUNCTION MGS_BaseFile::IsDimVar, name, position=position

   ;; Return if no dimensions were defined
   IF NOT Ptr_Valid(self.dims) THEN RETURN, 0

   IF N_Elements(name) NE 1 THEN BEGIN
      self->ErrorMessage, 'NAME must be a scalar string!'
      RETURN, 0
   ENDIF

   dimnames = StrLowCase( (*self.dims).name )
   position = Where(dimnames EQ StrLowCase(name), count)
   IF count GT 1 THEN BEGIN
      self->ErrorMessage, 'More than one dimension with name '+name+'!'
      retall
   ENDIF

   RETURN, (count GT 0)

END


; -----------------------------------------------------------------------------
; TestDimRange: (private)
;    This method analyzes a structure tag from the _Extra structure in
; GetData if its name corresponds to a dimension in the file. DimId is
; the dimension number of the variable (index in dims array and
; dimvars container), and value must adhere to the following rules:
; - if value is of type integer, it may contain 1-3 elements which are
;   interpreted as minindex, maxindex, stride. Integer compatible
;   types are also allowed (Byte, Long, UInt, ULong). Minindex and
;   maxindex are clipped to fall into the range 0..dims.value.
; - if value is of type float (or double) then it may contain 1 or 2
;   elements which are interpreted as minimum and maximum dimension
;   value. These are converted into indices for the dimension
;   variable.
; If these checks are successful, the value argument is converted into
; proper values for offset, count, and stride.
; Set the inclusive keyword if you want to be sure that the minimum
; and maximum value are included in the selection (this may lead to
; one or two extra values if, for example, the LAT dimension values
; are grid box centers and a point is requested that lies beyond the 
; center but within the same grid box). With the exclusive keyword
; set, the test is done as GT and LT. This is the default. A minimum
; of one value is always returned if value is not completely out of
; bounds. 

FUNCTION MGS_BaseFile::TestDimRange, dimid, value,   $
                     offset, count, stride,          $
                     inclusive=inclusive, exclusive=exclusive


   ;; Get number of value elements
   nval = N_Elements(value)
   
   ;; Return if outrageous
   IF nval GT 3 THEN BEGIN
      self->ErrorMessage, 'Too many values!'
      RETURN, 0
   ENDIF

   ;; Test for integer type
   Type = Size(value, /TNAME)
   isinteger = (Type EQ 'INT'  $
                OR Type EQ 'LONG' $
                OR Type EQ 'UINT' $
                OR Type EQ 'ULONG' $
                OR Type EQ 'LONG64' $
                OR Type EQ 'ULONG64' $
                OR Type EQ 'BYTE')
   isfloat = (Type EQ 'FLOAT' OR Type EQ 'DOUBLE')

;; ### DEBUG
; print,'### Testing dimension ',(*self.dims)[dimid].name
; print,value
; print,'isfloat, isinteger : ',isfloat,isinteger

   ;; Return if invalid type
   IF isinteger+isfloat EQ 0 THEN BEGIN
      self->ErrorMessage, 'Invalid value type!'
      RETURN, 0
   ENDIF

   ;; Make local working copy
   thisvalue = value

   ;; Get size of requested dimension
   dimsize = (*self.dims)[dimid].value

   ;; Handle float values
   IF isfloat THEN BEGIN
      ;; Return if more than two values
      IF nval GT 2 THEN BEGIN
         self->ErrorMessage, 'Too many values!'
         RETURN, 0
      ENDIF
      ;; If only one value is given, interprete as min and max 
      IF nval EQ 1 THEN thisvalue = [ thisvalue, thisvalue ]
      ;; Get dimension values
      thisdim = self.dimvars->Get(position=dimid)
      thisdim[0]->GetData, dimvals
      ;; Convert them to type of value
      typeid = Size(value, /TYPE)
      dimvals = FIX( dimvals, Type=typeid )
      ;; Make sure that size of dimvals corresponds to dims.value
      IF N_Elements(dimvals) NE dimsize THEN BEGIN
         self->ErrorMessage, 'Severe internal object error: Dimsize not consistent!'
         STOP
      ENDIF

      ;; Find maximum dimension value that is smaller or equal
      ;; value[0] and minimum dimension value that is greater or
      ;; equal value[1].
      ;; Special case: only one dimension value -> we are done
      IF N_Elements(dimvals) EQ 1 THEN BEGIN
         offset = 0L
         count = 1L
         stride = 1L
         RETURN, 1
      ENDIF
      ;; Test ordering of dimension values before proceeding
      isdescending = dimvals[0] GT dimvals[1]
      IF isdescending THEN dimvals = dimvals(Sort(dimvals))

      ;; ### NOTE: Special care should be taken here for circular
      ;; variables such as LON or LONGITUDE, etc. This will require an
      ;; extra subroutine or method ###
      wmin = Where(dimvals LE thisvalue[0], wcnt)
      IF Keyword_Set(inclusive) EQ 0 THEN wmin = wmin+1
      IF wcnt GT 0 THEN  $
        wmin = wmin[N_Elements(wmin)-1]  $
      ELSE  $
        wmin = 0L

      wmax = Where(dimvals GE thisvalue[1], wcnt)
      IF Keyword_Set(inclusive) EQ 0 THEN wmax = wmax-1
      IF wcnt GT 0 THEN  $
        wmax = wmax[0]  $
      ELSE $
        wmax = dimsize-1

      ;; For exclusive selection, return failure if value range lies
      ;; between two neighbouring dimension values - unless only one
      ;; value was given
      hasfailed = wmin GT wmax 
      
      ;; Need to 'invert' indices if dimension variable was descending
      IF isdescending THEN BEGIN
         tmp = wmin
         wmin = dimsize - wmax - 1
         wmax = dimsize - wmin - 1
      ENDIF
      
      ;; If only one value had been passed, but two values are found
      ;; as a result of the previous operations, take the one that is
      ;; closer. 
      IF nval EQ 1 AND wmin NE wmax THEN BEGIN
         diff0 = ABS(dimvals[wmin]-thisvalue[0])
         diff1 = ABS(dimvals[wmax]-thisvalue[0])
         IF diff0 LE diff1 THEN $
           wmax = wmin  $
         ELSE  $
           wmin = wmax
      ENDIF
      
      ;; Add a stride value of 1. The result is then equivalent to an
      ;; integer triple passed as value
      thisvalue = [ wmin, wmax, 1L ]

      IF hasfailed THEN RETURN, 0
   ENDIF

   ;; Handle integer values
   IF isinteger THEN BEGIN
      ;; first value must be positive or zero
      IF thisvalue[0] LT 0 THEN BEGIN
         self->ErrorMessage, 'First value must be GE 0!'
         RETURN, 0
      ENDIF
      ;; If only one value is given, interprete as min and max 
      IF nval EQ 1 THEN thisvalue = [ thisvalue, thisvalue ]
      ;; If second value is negative, replace with maximum possible
      ;; value
      IF thisvalue[1] LT 0 THEN thisvalue[1] = (*self.dims)[dimid].value-1
      ;; Make sure that second value does not exceed allowed value
      thisvalue[1] = thisvalue[1] < ((*self.dims)[dimid].value-1)
      ;; If no stride parameter is given, add 1 as stride
      IF nval LE 2 THEN $
        thisvalue = [ thisvalue, 1L ]
   ENDIF

   ;; Convert minindex, maxindex, stride to offset, count, and stride
   ;; minindex equals offset
   offset = thisvalue[0]
   ;; If stride is greater than max-min index, reset to 1
   idiff = thisvalue[1] - thisvalue[0]
   IF thisvalue[2] GT idiff THEN thisvalue[2] = 1L
   stride = thisvalue[2]
   ;; Compute count value
   count = Ceil( (idiff+1)/Float(stride) ) < ((*self.dims)[dimid].value)

; ### DEBUG
; print,'### offset,count,stride:',offset,count,stride

   RETURN, 1

END


; -----------------------------------------------------------------------------
; ReadVarData:
;    This method is a template for derived objects and needs to be
; overloaded for 'real world' file objects. It's purpose is to extract
; data for one variable from the file and limit the data to the
; dimension offset, count, and stride given in dims. If the derived
; file type does not support direct access to portions of the data, it
; must be read as a whole and stored in the object's vars variable
; (unless NoCopy is set), then the requested portion must be extracted
; using the Arrex function (see GetData for its use).

FUNCTION MGS_BaseFile::ReadVarData, data, variable=variable,  $
                dims=dims,   $
                NoCopy=nocopy, $
                _Extra=extra


   self->ErrorMessage, 'Generic file object cannot read data!'

   RETURN, 0

END


; -----------------------------------------------------------------------------
; BuildDataStructure:
;    This method creates a structure from a string array with tag
; names and a pointer array with the data values. In order to save
; time and memory, up to five tags are added to the structure in one
; step.

FUNCTION MGS_BaseFile::BuildDataStructure, tags, values

   ;; Get number of tags
   ntags = N_Elements(tags)

   ;; Create structure with first element
   result = Create_Struct(tags[0], *values[0])

   ;; Loop through remaining tags
   FOR i = 0L, (ntags-1)/5 DO BEGIN
      first = i*5+1
      last  = ((i+1)*5) < (ntags-1)
      count = last-first+1

      IF count EQ 1 THEN BEGIN
         result = Create_Struct(result, tags[first:last],*values[first])
      ENDIF ELSE IF count EQ 2 THEN BEGIN
         result = Create_Struct(result, tags[first:last],*values[first],  $
                                *values[first+1])
      ENDIF ELSE IF count EQ 3 THEN BEGIN
         result = Create_Struct(result, tags[first:last],*values[first],  $
                                *values[first+1],*values[first+2])
      ENDIF ELSE IF count EQ 4 THEN BEGIN
         result = Create_Struct(result, tags[first:last],*values[first],  $
                                *values[first+1], *values[first+2],  $
                                *values[first+3])
      ENDIF ELSE IF count EQ 5 THEN BEGIN
         result = Create_Struct(result, tags[first:last],*values[first],  $
                                *values[first+1], *values[first+2],  $
                                *values[first+3], *values[first+4])
      ENDIF
      ;; No else necessary.
   ENDFOR

   RETURN, result

END


; -----------------------------------------------------------------------------
; GetData:
;    This method is the main user interface for retrieving data from
; the file. You can choose variables by name or position index and
; limit each dimension by name and value (e.g. LON=0., LON=[0.,180.])
; or by name and index (integer instead of float values). In the
; latter case you can pass a third value which is interpreted as
; STRIDE value for this dimension (e.g. LON=[0,63,5] will extract
; every fifth longitude value starting from index zero and not
; extending beyond index 63).
;    Per default, GetData returns an object array with
; MGS_Basevariable objects. Set the AsStructure keyword if you prefer
; to retrieve a structure instead.
;    If an error occurs, the method returns an empty object (test with
; Obj_Valid()) or -1L if the AsStructure keyword is set (test with
; ChkStru()). 
;
; The method searches the variable list for matching names and loops
; over these variables to perform the following operations:
; 1. Prepare the variable dimensions, count, offset, and stride values
; 2. If (complete) variable was loaded previously, extract the
;    requested portion from memory, otherwise retrieve requested data
;    portion from file. If no range limit applies, load the complete
;    variable and store it in the variable object for later use
;    (unless the NoCopy keyword is set)
; 3. Build a new variable object that copies the original one and
;    contains the potentially truncated dimensions.
; 4. After all variables have been retrieved, check which dimensions
;    they need and add the respective dimension variables to the
;    variable object array (unless the NoDimensions) keyword is set.
; Notes:
; (1) Be careful when limiting dimensions that you choose the correct
; data type! Float values are interpreted as dimension values whereas
; integer values are interpreted as dimension indices!
; (2) Derived objects should only overload this method if the file
; type does not support retrieval of individual variables from the
; file. Otherwise modifications should be restricted to the
; ReadVarData method.
; (3) Do not use _Ref_Extra in this method! Range selection is only
; possible by exploiting the fact that the _Extra keyword variable is
; a plain IDL structure. It would not work with _Ref_Extra.
; (4) Position overrules Variables if both keywords are used (this is
; a property of the MGS_Container object).
; (5) Normally, the variables will be arranged in the order they are
; specified, dimensions appended at the end. Set the PreserveOrder
; keyword if you want the variables to be returned in the order they
; are stored in the file.
;
; Examples:
;   thisvar = thefile->GetData(variable='nox')
;   ; reads all data for the variable named NOX and returns an object
;   ; array with one variable for NOX and one variable for each
;   ; dimension of NOX.
;
;   thisvar = thefile->GetData(variable='O*', lon=[-20.,60.], $ 
;                              lat=[20.,80.], lev=30) 
;   ; returns all variables whose name start with 'O' and limits them
;   ; to Europe (20 deg W to 60 deg E, and 20 deg N to 80 deg N). The
;   ; 31st level is extracted regardless of its value.
;
;   thisdata = thefile->GetData(variable=['O3','CO'],  $
;                               time=[3,9999,7], /AsStructure)
;   ; returns a structure containing O3 and CO data for every 7th time
;   ; step beginning from step 4.
;
;   thisdata = thefile->GetData(position=[2,5,9], /AsPointer)
;   ; returns an object array(!) with variables number 2, 5, 
;   ; and 9. The Data fields of the variable objects are pointers to
;   ; the same data that are stored in the file object's variables.
;

FUNCTION MGS_BaseFile::GetData, Variables=variables, $
                     Position=position,  $
                     AsStructure=asstructure,  $
                     NoCopy=nocopy,  $
                     NoDimensions=nodimensions,  $
                     PreserveOrder=preserveorder,  $
                     DEBUG=DEBUG,  $
                     _Extra=extra

; *** TEMPORARY ***** (because exclusive doesn't always do what i want)
inclusive = 1

   ;; Set dummy return value
   IF Keyword_Set(asstructure) THEN $
     dummy = -1L  $
   ELSE  $
     dummy = Obj_New()

   thisvarname = ''

   ;; Error handler
   CATCH, theError
   IF Keyword_Set(DEBUG) THEN CATCH,/CANCEL
   IF theError NE 0 THEN BEGIN
      CATCH, /Cancel
      self->ErrorMessage, 'Error retrieving variable '+thisvarname+'!'
      RETURN, dummy
   ENDIF

   ;; Return if file object contains no variables or no dimensions
   IF NOT Obj_Valid(self.vars) OR NOT Obj_Valid(self.dimvars) THEN BEGIN
      self->ErrorMessage, 'File object not properly initialized!', /Warning
      RETURN, dummy
   ENDIF

   nvars = self.vars->Count()
   ndims = self.dimvars->Count()

   ;; Open file if not open and no variables or dimensions stored
   IF nvars EQ 0 AND ndims EQ 0 AND NOT self->IsOpen() THEN BEGIN
      self->Open
      nvars = self.vars->Count()
      ndims = self.dimvars->Count()
   ENDIF

   IF nvars EQ 0 THEN BEGIN
      self->ErrorMessage, 'File object contains no variables!', /Warning
      RETURN, dummy
   ENDIF

   IF NOT Obj_Valid(self.dimvars) THEN BEGIN
      self->ErrorMessage, 'File object contains no dimension variables!', $ 
        /Warning
      RETURN, dummy
   ENDIF

   ;; Initialize diminfo structure array (used for subrange
   ;; extraction) 
   diminfo = Replicate( { offset:0L, count:1L, stride:1L}, ndims )

   ;; preset count values with dimension size values
   diminfo.count = (*self.dims).value
   IF Keyword_Set(DEBUG) THEN BEGIN
      print,'---initial---'
      print,'DIMNAMES:',(*self.dims).name
      print,'OFFSET:',diminfo.offset
      print,'COUNT:',diminfo.count
      print,'STRIDE:',diminfo.stride
   ENDIF

   ;; Check _Extra keyword structure if it contains valid dimension
   ;; names. Find respective dimension IDs and compute offset, count,
   ;; and stride values for each dimension.
   IF ChkStru(extra) THEN BEGIN
      islimited = 1

      ;; Loop over structure tags and extract valid dimension limits.
      extranames = StrUpCase(Tag_Names(extra))

      FOR i=0L, N_Tags(extra)-1 DO BEGIN
         ;; Is tag name the name of a dimension?
         wdim = Where( StrUpCase((*self.dims).name) EQ extranames[i], wcnt )
         IF wcnt GT 0 THEN BEGIN
            IF self->TestDimRange(wdim[0], extra.(i), $
                                  offset, count, stride,  $
                                  inclusive=inclusive) THEN BEGIN
               diminfo[wdim[0]].offset = offset
               diminfo[wdim[0]].count  = count
               diminfo[wdim[0]].stride = stride
            ENDIF ELSE BEGIN
               self->ErrorMessage, 'Invalid dimension limit for '+extranames[i]+'!'
               RETURN, dummy
            ENDELSE
         ENDIF   
         ;; Ignore other tags (may be passed to ReadVarData method of 
         ;; derived objects)
      ENDFOR

      IF Keyword_Set(DEBUG) THEN BEGIN
         print,'---final---'
         print,'DIMNAMES:',(*self.dims).name
         print,'OFFSET:',diminfo.offset
         print,'COUNT:',diminfo.count
         print,'STRIDE:',diminfo.stride
      ENDIF
   ENDIF ELSE BEGIN

      islimited = 0

   ENDELSE


   ;; Set up object array to store the retrieved data as variable
   ;; objects. If a structure is requested it will be formed later
   ;; from the objects.
   ;; Note: At this point, result and the objects in self.vars point
   ;; to the same variable structures.
   thevars = self.vars->Get(name=variables, position=position, count=count, $
                            sorted=Keyword_Set(preserveorder))

   ;; If no variables match the request, return
   IF count EQ 0 THEN RETURN, dummy

   ;; Accumulate dimension indices that are used so that dimension
   ;; variables can be added later
   isdimused = IntArr(ndims)
   isdimrequested = IntArr(ndims)

   ;; Define temporary storage (pointers) if data shall be returned as
   ;; structure
   IF Keyword_Set(AsStructure) THEN BEGIN
      strutags = StrArr(count)
      strudata = PtrArr(count)
   ENDIF

   ;; Loop over variables and retrieve data
   FOR i=0L, count-1 DO BEGIN
      
      ;; 1. Map dimensions
      ;; Copy values from diminfo that are relevant for current
      ;; variable and test if current variable is a dimension variable
      ;; itself
      thevars[i]->GetProperty, name=thisname, ndims=nvdims
      IF self->IsDimVar(thisname, position=position) THEN $
         isdimrequested[position] = 1

      IF nvdims GT 0 THEN BEGIN
         thesedims = LonArr(nvdims)
         FOR j=0,nvdims-1 DO BEGIN
            thevars[i]->GetDimVar, j+1, thisdimid
            thesedims[j] = thisdimid
            isdimused[thisdimid] = 1
         ENDFOR
         IF Keyword_Set(DEBUG) THEN $
            print,'Variable '+thisname+' has dimension indices:',thesedims
      ENDIF

      ;; 2. Test if data already resides in memory
      ;; If so, extract requested range and store as local temporary 
      ;; variable 
      ;; *** Note: the mgs_variable object should support arrex
      ;; directly so that no extra copy of the data required! ***
      IF thevars[i]->ContainsData() THEN BEGIN
         thevars[i]->GetData, tmpdata
         IF nvdims GT 0 AND islimited THEN BEGIN
            tmpdata = ArrEx(tmpdata,  $
                            start=diminfo[thesedims].offset, $
                            count=diminfo[thesedims].count,  $
                            stride=diminfo[thesedims].stride)
         ENDIF
      ENDIF ELSE BEGIN
         ;; Data not in memory, need to load from file
         ;; First, make sure file is open
         IF NOT self->isOpen() THEN self->ReOpen
         ;; Extraction of subrange will be done in ReadVarData
         thevars[i]->getproperty,name=thename
         isok = self->ReadVarData(tmpdata, variable=thevars[i], $
                                  dims=diminfo[thesedims], $
                                  NoCopy=nocopy, _Extra=extra)
         IF NOT isok THEN BEGIN
            self->ErrorMessage, 'Cannot retrieve variable from file!'
            GOTO, cycle
         ENDIF

         ;; If full variable was retrieved, store data in variable
         ;; object unless nocopy keyword is set 
         IF NOT islimited AND NOT Keyword_Set(nocopy) THEN $
            thevars[i]->SetProperty, data=tmpdata
      ENDELSE
      
      ;; 3. Create a physical copy of current variable and store the
      ;; data in it or add data to temporary structure buffer
      IF Keyword_Set(AsStructure) THEN BEGIN
         thevars[i]->GetProperty,name=thename
         strutags[i] = thename
         strudata[i] = Ptr_New(tmpdata,/No_Copy)
      ENDIF ELSE BEGIN
         thevars[i] = thevars[i]->Copy(/Free_Pointers,/Free_Objects)
         thevars[i]->SetProperty, Data=tmpdata
         ;; Add names of dimension variables
         FOR j=0L,nvdims-1 DO BEGIN
            thevars[i]->SetDimVar, j+1, (*self.dims)[thesedims[j]].name
         ENDFOR
      ENDELSE

cycle:
   ENDFOR

   ;; Add dimension variables that were not explicitely listed unless
   ;; the NoDimensions keyword is set
   IF Keyword_Set(NoDimensions) EQ 0 THEN BEGIN
      ;; Find out which dimensions are used by requested variables but
      ;; have not been explicitely requested
      wdim = Where(isdimused*(1-isdimrequested), wcnt)
      IF wcnt GT 0 THEN BEGIN
         extravars = self->GetData(variables=(*self.dims)[wdim].name,  $
                                   AsStructure=asstructure,  $
                                   PreserveOrder=preserveorder, Debug=debug,  $
                                   _Extra=extra)
      ENDIF
   ENDIF

   ;; Form result: build a structure or return variable objects
   IF Keyword_Set(AsStructure) THEN BEGIN
      result = self->BuildDataStructure(strutags, strudata)
      IF N_Elements(extravars) GT 0 THEN $
         result = Create_Struct(result, extravars)
   ENDIF ELSE BEGIN
      result=thevars
      IF N_Elements(extravars) GT 0 THEN $
         result = [ result, extravars ]
   ENDELSE


   RETURN, result
END


; -----------------------------------------------------------------------------
; Read:
;    This method is meant to scan all data headers and read all data
; from the file. Use the HeaderOnly keyword to read only the header
; information. 
;    In the generic file object this method issues a warning message
; since it cannot be known how the file contents must be
; handled. Derived objects must overload this method.
; Notes:
; (1) As a user you will generally access data via the GetData method
; rather than calling Read and GetProperty, vars=vars. The Read method
; is primarily designed for internal purposes.

PRO MGS_BaseFile::Read,   $
                HeaderOnly=headeronly,  $
                _Ref_Extra=e

   self->ErrorMessage, 'Generic file object cannot read data!'

END


; -----------------------------------------------------------------------------
; IsOpen:
;    This method tests if a file has been opened. In this generic file
; object, only the value of fileID is tested. Derive objects should
; perform a 'physical' test before resetting the fileID value.

FUNCTION MGS_Basefile::IsOpen

   RETURN, self.fileID GE 0

END


; -----------------------------------------------------------------------------
; OpenAgain:  (private)
;   For the generic file object, this method is a dummy
; method. Derived objects may overwrite this method to close the file
; that was previously opened with Open_File as a simple text or raw
; binary file and reopen it with a special command or extra keywords
; for the Open_File procedure.

FUNCTION MGS_Basefile::OpenAgain,   $
                     lun,  $
                     _Extra=e

   ;; Dummy. Simply return the logical unit number
   RETURN, lun

END


; -----------------------------------------------------------------------------
; Open:
;    This method provides a generic user interface to open existing
; files for reading or changing. First, a file matching the search
; pattern of the object's filename attribute is located using the
; Open_File procedure. This procedure returns a logical unit number or
; -1 if the attempt to open it as a raw binary or simple text file has
; failed (see documentation on Open_File). Afterwards, the OpenAgain
; method is called with the logical file unit as argument. 
; Instead of overloading the Open method, derived objects should
; change OpenAgain to accomodate for special opening keywords or
; different file open commands (e.g. netcdf). The result of the
; OpenAgain method is then stored in fileId. 
;    Normally, the object's filename field is used as a search pattern
; for Open_File. The filename argument can be used to overwrite this,
; for example to re-open the file with the previously determined exact
; file name (see ReOpen method).
;    Unless the NoScan keyword is set, a call to the object's Read method
; with the HeaderOnly keyword will automatically load the header
; information (metadata) for subsequent access to the data. This is
; done even if the file is opened for writing in case you want to
; change its contents.  
; Notes:
; (1) This method is not suited for creating a new file (even
; though this may work under some circumstances). Use the Create
; method for this purpose.
; (2) FORTRAN 77 unformatted files and byte swapping can be achieved
; by passing the respective keywords directly to this method. Yet, a
; derived file object should overload the OpenAgain method to open a
; FORTRAN binary file to increase transparency to the user.
; (3) Use the no_pickfile keyword if you are batch processing and
; don't want to interrupt program flow.
; (4) _Extra keywords are stored in the object for subsequent use by
; the ReOpen method.

PRO MGS_Basefile::Open,  $
                Filename=filename,  $
                NoScan=noscan, $; prevent scanning of data headers
                Result=result, $; flag to indicate success or failure
                Write=write, $  ; do not allow write keyword for open_file
                Update=update, $; same for update mode if readonly flag is set
                _Extra=e        ; for derived object's keywords


   result = 0

   ;; Return if file is already open
   IF self->IsOpen() THEN RETURN

   ;; Reset extraopen field
   IF Ptr_Valid(self.extraopen) THEN Ptr_Free, self.extraopen 

   ;; Set default for filename
   IF N_Elements(filename) EQ 0 THEN filename = self.filename

   ;; Attempt to open file as text or raw binary file
   title = 'Select '+self.filetype+' file:'
   Open_File, filename, lun, filename=truename, title=title,  $
     Update=(self.readonly EQ 0), _EXTRA=e

   ;; Save exact filename
   IF lun GE 0 THEN self.truename = truename

   ;; Re-Open the file in the special mode required by specific file
   ;; type objects and set file id
   self.fileID = self->OpenAgain(lun, _EXTRA=e)
   
   ;; Return result
   result = (self.fileID GE 0)

   ;; Store _extra keywords for later use (only if successful)
   IF result AND N_Elements(e) GT 0 THEN self.extraopen=Ptr_New(e)

   ;; Scan headers
   IF Keyword_Set(noscan) EQ 0 THEN $
     self->Read, /HeaderOnly, _Extra=e

END


; -----------------------------------------------------------------------------
; ReOpen: 
;   This method opens a file that has been opened and scanned
; previously. It uses the truename field to provide the Open method
; with an exact filename and calls Open with the NoScan keyword.
; This method is automatically called if data shall be retrieved from
; a closed file.
;   Extra keywords used at the first call to open are reused.
; Unless the noscan keyword is set, all variable and dimension
; information is scanned again (safety line).

PRO MGS_Basefile::ReOpen, result=result, noscan=noscan

   ;; Get extra keywords from first open command
   IF Ptr_Valid(self.extraopen) THEN $
     e = *self.extraopen  $
   ELSE  $
     e = { nothing:-1L }

   ;; Use truename as filename unless it is empty
   filename = self.truename
   IF filename EQ '' THEN filename = self.filename

   self->Open, filename=filename, result=result,  $
     noscan=noscan, _EXTRA=e

END


; -----------------------------------------------------------------------------
; Create:
;    This method creates a new file with the name stored in the object
; as filename. The Open_File procedure is called with the Write
; keyword to locate the new file (in case filename contains a search
; pattern) and to create a new text or raw binary file. Derived
; objects may use the result from this operation to test if a file
; with this name can be created and then recreate the file with their
; specific create procedure.

PRO MGS_BaseFile::Create, _Extra=e

   ;; Return if the file is marked read only or already open
   IF self.readonly THEN BEGIN
      self->ErrorMessage, 'Cannot create a file without write access allowed!'
      RETURN
   END
   IF self->IsOpen() THEN BEGIN
      self->ErrorMessage, 'Cannot create an already open file!'
      RETURN
   END

   ;; Reset truename and extraopen fields
   self.truename = ''
   IF Ptr_Valid(self.extraopen) THEN Ptr_Free, self.extraopen

   ;; Call Open_File
   Open_File, self.filename, lun, filename=truename, /WRITE, $
     title='Enter a filename or select a file:', _Extra=e

   ;; Enter logical unit number as file id and save exact filename
   IF lun GE 0 THEN BEGIN
      self.fileid = lun
      self.truename = truename
      IF N_Elements(e) GT 0 THEN self.extraopen = Ptr_New(e)
   ENDIF
END


; -----------------------------------------------------------------------------
; Close:
;    This method closes the file. It frees the logical unit number
; stored in fileID and sets fileID to -1. Derived objects may have
; to overload this method if they require special close procedures
; (e.g. netcdf).

PRO MGS_Basefile::Close

   ;; Close the file
   IF self.fileID GE 0 THEN $
     Free_Lun, self.fileID

   ;; Mark file as closed
   self.fileID = -1

END


; -----------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user. Information about variables or dimensions can be retrieved
; more comfortably with the Get... methods.
;   If the gattr, dims, or vars properties are not filled with valid
; information they return undefined variables (check with N_Elements()
; GT 0).
; Note: 
;    Unlike GetVariable, this method will not copy the variable
; objects but return pointers to the vaiables stored in this object
; instead. Do not destroy the variable objects that you retrieve this
; way! For dimensions and attributes, a local copy is made.


PRO MGS_Basefile::GetProperty, $
   Filename=filename, $       ; The current true filename (i.e. truename!)
   SearchPattern=searchpattern, $ ; The file search pattern (i.e. filename)
   FileId=fileid,    $        ; The file ID  (debug purposes only !!)
   FileType=filetype, $       ; The file type
   ReadOnly=readonly, $       ; File status flag
   WriteAccess=writeaccess, $ ; Same negated
   GAttr=gattr,  $            ; The global attributes (structure) of the file
   Title=title, $             ; The global title attribute of the file
   Dims=dims,  $              ; The info about the dimensions (name, size, range)
   DimVars=dimvars,  $        ; An object array with dimension variables
   Variables=vars,  $         ; An object array with variables
   Varnames=varnames,  $      ; The variable names
   _Ref_Extra=extra           ; Inherited and future keywords
                              ;
                              ; Inherited keywords:
                              ; name      : The variable name
                              ; uvalue    : a user-defined value


   ;; Get properties from base object
   self->MGS_BaseObject::GetProperty, _Extra=extra

   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving object properties!'
      RETURN
   ENDIF

   ;; Get file properties
   filename = self.truename
   searchpattern = self.filename
   fileid = self.fileid
   filetype = self.filetype
   readonly = self.readonly
   writeaccess = 1-self.readonly

   ;; Get data properties
   title = self.title

   IF Arg_Present(gattr) THEN BEGIN
      IF Ptr_Valid(self.gattr) THEN $
        gattr = *self.gattr  $
      ELSE  $
        self->Undefine, gattr
   ENDIF

   IF Arg_Present(dims) Then BEGIN
      IF Ptr_Valid(self.dims) THEN $
        dims = *self.dims $
      ELSE  $
        self->Undefine, dims
   ENDIF

   IF Arg_Present(dimvars) THEN BEGIN
      IF self.dimvars->Count() GT 0 THEN $
        dimvars = self.dimvars->Get(/All)  $
      ELSE $
        self->Undefine, dimvars
   ENDIF

   IF Arg_Present(vars) THEN BEGIN
      IF self.vars->Count() GT 0 THEN $
        vars = self.vars->Get(/All)  $
      ELSE $
        self->Undefine, vars
   ENDIF

   IF Arg_Present(varnames) THEN BEGIN
      IF self.vars->Count() GT 0 THEN $
        varnames = self.vars->GetNames()  $
      ELSE $
        self->Undefine, varnames
   ENDIF

END 


; -----------------------------------------------------------------------------
; SetProperty:
; This method sets specific object values. Derived objects may want to
; overwrite and extend this method to allow storing additional
; information.
; In order to ensure internal consistency, certain restrictions apply:
;    The filename and the readonly (writeaccess) flag may only be
; changed if the file is closed. Use writeaccess=0 to protect a file
; from writing that may have had write access earlier.
;    The vars, dims, and gattr fields may only be changed if the
; readonly flag allows writing. Warning: changing these fields causes
; the entire file content to be replaced and may lead to data loss!!
; Only minimal consistency checks are made on dims, gattr, and vars.
; Note:
;    If vars are provided to replace the current variable definitions, the
; old container is destroyed. This causes all variable objects stored
; in the container to be destroyed as well and may lead to invalid
; object references if variables have been extracted from the object
; with the NoCopy keyword.

PRO MGS_Basefile::SetProperty, $
   Filename=filename, $            ; The file name or file mask of a netcdf file
                                   ; (used for next OPEN attempt)
   WriteAccess=writeaccess, $      ; Unprotect file from writing
   Dims=dims,  $
   DimVars=dimvars,  $
   Gattr=gattr,  $
   Vars=vars,  $
   _Ref_Extra=extra                ;
                                   ; Inherited keywords:
                                   ; name      : The variable name
                                   ; no_copy   : Don't keep local copy
                                   ;             of uvalue
                                   ; no_dialog : Don't display
                                   ;             interactive dialogs
                                   ; uvalue    : a user-defined value


   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error setting object properties!'
      RETURN
   ENDIF

   ;; Set Properties of base object
   self->MGS_BaseObject::SetProperty, _Extra=extra


   ;; Apply changes to filename and readonly if file is closed
   IF N_Elements(filename) NE 0 THEN $
      IF NOT self->IsOpen() THEN self.filename = filename

   IF N_Elements(writeaccess) GT 0 THEN $
     IF NOT self->IsOpen() THEN self.readonly = 1-Keyword_Set(writeaccess)

   ;; Apply changes to data description if file allows writing
   ;; Side effect: title value will be change if gattr structure
   ;; contains a title tag.
   IF NOT self.readonly THEN BEGIN

      ;; Global attributes
      IF N_Elements(gattr) GT 0 THEN BEGIN
         IF Ptr_Valid(self.gattr) THEN Ptr_Free, self.gattr
         self.gattr = Ptr_New(gattr)  
         IF ChkStru(gattr, 'title') THEN self.title = gattr.title
      ENDIF 
   
      ;; Dimensions
      IF N_Elements(dims) GT 0 THEN BEGIN
         IF NOT ChkStru(dims[0], ['name', 'size', 'range']) THEN BEGIN
            self->ErrorMessage, ['Dims keyword must be structure array!', $
                                 'Structure elements must contain "name", '+ $
                                 '"size", and "range" fields.']
         ENDIF
         IF Ptr_Valid(self.dims) THEN Ptr_Free, self.dims
         self.dims = Ptr_New(dims) 
      ENDIF
   
      ;; Variables and dimension variables:
      ;; Destroy old variable container rather than removing it's
      ;; entries so that memory gets freed. Delete vars or dimvars
      ;; only if they are not also stored in the other container. 
      savedimvars = 1
      IF N_Elements(dimvars) GT 0 THEN BEGIN
         vtype = Size(dimvars, /TName)
         IF vtype NE 'OBJREF' THEN BEGIN
            self->ErrorMessage, 'DimVars keyword must be variable object list!'
         ENDIF
         IF Obj_Valid(self.dimvars) THEN BEGIN
            ;; Check for cross references
            ;; Remove cross referenced dimension variables from container
            IF Obj_Valid(self.vars) THEN BEGIN
               ndv = self.dimvars->Count()
               FOR i=ndv-1,0,-1 DO BEGIN
                  IF self.vars->IsContained(self.dimvars->Get(position=i)) THEN $
                    self.dimvars->Remove, position=i
               ENDFOR
            ENDIF
            Obj_Destroy, self.dimvars
            self.dimvars = Obj_New('MGS_Container')
            savedimvars = 0
         ENDIF
         FOR i=0,N_Elements(vars)-1 DO $
           self.vars->Add, vars[i]
      ENDIF


      IF N_Elements(vars) GT 0 THEN BEGIN
         vtype = Size(vars, /TName)
         IF vtype NE 'OBJREF' THEN BEGIN
            self->ErrorMessage, 'Vars keyword must be variable object list!'
         ENDIF
         IF Obj_Valid(self.vars) AND savedimvars THEN BEGIN
            ;; Check for cross references
            ;; Remove cross referenced variables from container
            IF Obj_Valid(self.dimvars) THEN BEGIN
               nv = self.vars->Count()
               FOR i=nv-1,0,-1 DO BEGIN
                  IF self.dimvars->IsContained(self.vars->Get(position=i)) THEN $
                    self.vars->Remove, position=i
               ENDFOR
            ENDIF
            Obj_Destroy, self.vars
            self.vars = Obj_New('MGS_Container')
         ENDIF
         FOR i=0,N_Elements(vars)-1 DO $
           self.vars->Add, vars[i]
      ENDIF
   ENDIF

END 


; -----------------------------------------------------------------------------
; Cleanup:
; This method closes the file and frees all data stored in the object.
; Unusual for a cleanup method but maybe useful, the Keep_Variables
; keyword prevents destruction of the file's data contents. You must
; retrieve the gattr, dims, vars, and dimvars fields with the
; GetProperty method before destroying the object, otherwise you will
; have a memory leak. The gattr and dims pointers will be freed anyhow
; because they are returned as values by the GetProperty method.

PRO MGS_Basefile::CLEANUP, KeepVariables=Keep_Variables

   ;; Destroy any GUI
   ;; self->KillGUI

   ;; Close the file
   self->Close

   ;; Delete extraopen information
   IF Ptr_Valid(self.extraopen)  THEN Ptr_Free, self.extraopen
   
   ;; Destroy file content information
   IF Ptr_Valid(self.gattr)    THEN Ptr_Free, self.gattr
   IF Ptr_Valid(self.dims)     THEN Ptr_Free, self.dims
   IF Keyword_Set(keepvariables) EQ 0 THEN BEGIN
      IF Obj_Valid(self.vars)     THEN Obj_Destroy, self.vars
      IF Obj_Valid(self.dimvars)  THEN Obj_Destroy, self.dimvars
   ENDIF

   ;; Call parent's cleanup method
   self->MGS_BaseObject::Cleanup

END 




; -----------------------------------------------------------------------------
; Init:
;   This method initializes the file object. It does not open the file
; or check it's existance.
;   You can use the vars, dims, dimvars, and gattr keywords to
; initialize the file content variables (e.g. from another object
; accessing the same file). However, only few consistency checks are
; made for these keywords, and it is generally safer to use the
; AddDimension, AddGattr, and AddVariable methods to define the data
; that shall be stored in the file. If a file will be opened for reading
; (WriteAccess flag not set), the vars, dims, and gattr keywords are
; ignored. 
;    Derived objects should overload this method to set the filetype
; value correctly. They may also choose to use the defaultmask keyword
; in the initialization of the parent object (which is this object).
; Notes:
; (1) The value of the readonly flag is not necessarily constant
; throughout the lifetime of the file object, but may be changed with
; the SetProperty method if the file is not open. 
; (2) Dimvars may contain references to objects that are stored in vars.


FUNCTION MGS_Basefile::INIT, $
                     Filename=filename, $ ; The name of the file or a
                                ; search pattern to be used in the
                                ; file open dialog.
                     WriteAccess=writeaccess, $ ; A flag to allow write access 
                                ; when opening the file
                     Title=title,  $      ; A title string for widgets
                                ; describing the contents of the
                                ; file. Normally, this information is
                                ; extracted from the title tag of the
                                ; gattr structure.
                     Vars=vars, $         ; An object array with variables 
                     Dims=dims,  $        ; A structure array with dimension
                                ; information
                     DimVars=dimvars, $   ; An object array with variable 
                                ; objects storing the dimension data
                     Gattr=gattr, $       ; A structure with global attributes
                     DefaultMask=defaultmask,  $  ; A default search pattern
                                ; to be used if no filename is given (private)
                     _Ref_Extra=extra     ; For inherited and future keywords
                                ;
                                ; Inherited keywords:
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value



   ;; Initialize parent object first
   IF not self->MGS_BaseObject::Init(_Extra=extra) THEN RETURN, 0
   
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error initializing object!'
      RETURN, 0
   ENDIF
   
   ;; Set filetype property and readonly flag
   self.filetype = '(Unknown type)'
   self.readonly = 1-Keyword_Set(writeaccess)

   IF N_Elements(filename) EQ 0 THEN filename = ''
   ;; Check filename keyword
   self.filename = StrTrim(filename,2)
   
   ;; Replace empty filename with default mask ('*')
   ;; NB: Always use '*' on Windows systems because mask cannot be
   ;; altered
   IF self.filename EQ '' THEN BEGIN
      IF StrUpCase(!Version.OS_Family) EQ 'WINDOWS' THEN BEGIN
         self.filename = '*'  
      ENDIF ELSE BEGIN
         IF N_Elements(defaultmask) GT 0 THEN BEGIN
            self.filename = defaultmask
         ENDIF ELSE BEGIN
            self.filename = '*'
         ENDELSE
      ENDELSE
   ENDIF

   ;; Set file attributes
   self.truename = ''
   self.fileid = -1L
   self.extraopen = Ptr_New()

   ;; Set title: Title keyword is overwritten if gattr contains a
   ;; title tag.
   IF N_Elements(title) GT 0 THEN $
     self.title = StrTrim(title, 2)  $
   ELSE $
     self.title = ' '
   
   ;; Set data attributes.
   ;; This section is skipped if the file is readonly
   self.vars = Obj_New('MGS_Container') 
   self.dimvars = Obj_New('MGS_Container')

   IF self.readonly EQ 0 THEN BEGIN
      self.gattr = Ptr_New()
      self.dims = Ptr_New()
   ENDIF ELSE BEGIN
      IF N_Elements(gattr) GT 0 THEN BEGIN
         self.gattr = Ptr_New(gattr)  
         IF ChkStru(gattr, 'title') THEN self.title = gattr.title
      ENDIF ELSE $
        self.gattr = Ptr_New()
      
      IF N_Elements(dims) GT 0 THEN BEGIN
         IF NOT ChkStru(dims[0], ['name', 'size', 'range']) THEN BEGIN
            self->ErrorMessage, ['Dims keyword must be structure array!', $
                                 'Structure elements must contain "name", '+ $
                                 '"size", and "range" fields.']
            RETURN, 0
         ENDIF
         self.dims = Ptr_New(dims) 
      ENDIF ELSE $
        self.dims = Ptr_New()
      
      IF N_Elements(vars) GT 0 THEN BEGIN
         vtype = Size(vars, /TName)
         IF vtype NE 'OBJREF' THEN BEGIN
            self->ErrorMessage, 'Vars keyword must be variable object list!'
            RETURN, 0
         ENDIF
         FOR i=0,N_Elements(vars)-1 DO $
           self.vars->Add, vars[i]
      ENDIF

      IF N_Elements(dimvars) GT 0 THEN BEGIN
         vtype = Size(dimvars, /TName)
         IF vtype NE 'OBJREF' THEN BEGIN
            self->ErrorMessage, 'DimVars keyword must be variable object list!'
            RETURN, 0
         ENDIF
         FOR i=0,N_Elements(dimvars)-1 DO $
           self.dimvars->Add, dimvars[i]
      ENDIF

   ENDELSE
   
   RETURN, 1
END 



; -----------------------------------------------------------------------------
; MGS_Basefile__Define:
; This is the object definition for a generic file object. It
; inherits from MGS_BaseObject the abilities to set and query an
; object name and a uvalue. The base object also provides a general
; method for display of error messages which can be directed to a
; message dialog or to the log screen via the no_dialog flag.

PRO MGS_Basefile__Define

   objectClass = { MGS_Basefile, $ ; The object class
           ;;; file information
           filename: '', $    ; Name of file or search pattern
           truename: '', $    ; Fully qualified file name after opening
           filetype: '', $    ; A file type identifier for widgets etc.
           fileid: -1L,  $    ; File ID (-1 when closed)
           readonly: 1,  $    ; Flag indicating access mode
           extraopen: Ptr_New(),$  ; Extra information for reopening file
           ;;; data information
           title: '',  $      ; A data set describing title
           gattr: Ptr_New(),$ ; Structure array with global attributes
           dims: Ptr_New(), $ ; Structure array with dimension info
           dimvars: Obj_New(),$ ; Container with dimension variable objects
           vars: Obj_New(), $ ; Container with variable objects
           ;;; heritage
           INHERITS MGS_BaseObject  $ ; provides basic general properties
     }

END
