;+
; NAME:
;     mgs_contour (procedure)
;
; PURPOSE:
;     This is a procedure interface for the MGS_Contour object.
;
; ARGUMENTS:
;     DATA -> A 2-dimensional array with the data values. Required
;        unless the variable keyword is used.
;
;     X, Y -> The x and y values (or lons and lats) of data. If either
;        one is not present or set to a scalar, the index values
;        of data will be used instead.
;
; KEYWORDS:
;     LEVELS (floatarray, input) -> The data values for which contour
;        levels shall be drawn. Note that for filled contours the first
;        value of levels is the upper limit of the first
;        interval. ???? If not provided, levels will be determined
;        automatically using the MINVALUE and MAXVALUE keyword or the
;        minimum and maximum of the data within the specified LIMIT.
;
;     NLEVELS (integer, input) -> If NLEVELS are not given, use this
;        keyword to specify the number of contour levels. If not given,
;        the program will compute levels automatically to yield
;        approximately 15 levels.
;
;     MINVALUE, MAXVALUE (float, input) -> If LEVELS are not given,
;        they are determined automatically. Use this keyword to
;        specify the range of contour lines.
;
;     LOG (boolean, input) -> Set this flag to indicate logarithmic
;        spacing of contour levels.
;
;     DRAWCOLOR (string or long or integer, input) -> Drawing color for
;        axes etc. Default is "black". If color is a string value,
;        David Fanning's GetColor program is used to create a device
;        independent color value. Otherwise, the user must know
;        whether color is an index to the color table (if the device
;        is not decomposed) or a long value containing the RGB
;        information (if decomposed=1). The drawing color is loaded to
;        the last table index but one.
;
;     BACKCOLOR (string or long or integer, input) -> Background
;        color. Default is "white". See COLOR keyword for details.
;
;     C_COLORS (integerarray, input) -> The color indices (??? true
;        color values ???) for the contour lines or filled
;        contours. Default is to use NLEVELS colors beginning from 1.
;
;     CTABLE (integer, input) -> The index of the color table to
;        be used for contours or image display. Default is 27
;        (EOS-B). The number of colors loaded depends on the number of
;        levels defined (default is nlevels+1=16).
;
;     MAP (boolean, input) -> Set this flag to indicate that data is
;        geographical and should be displayed on a lon-lat map.
;
;     LIMIT (4-integers, input) -> Specify the extend in x and y that
;        shall be displayed. As with MAP_SET, the 4 elements of LIMIT
;        must be arranged as [lat0, lon0, lat1, lon1]. Note that LIMIT
;        can be used for non-geographical data as well. If X is a
;        longitude coordinate (MAP keyword set), lon1 may be smaller
;        than lon0, in which case the data are wrapped.
;
;     GRID_FILE (string, input) -> Name of a Netcdf file containing
;        the longitude and latitude values of the data (Variables must
;        be named 'lon' or 'longitude', and 'lat' or 'latitude'
;        (case-insensitive). The filename may contain wildcards ('*',
;        '?') in which case the user can select the file
;        interactively. Use of GRID_FILE implies automatic setting of
;        MAP.
;
;     LINES (boolean, input) -> Draw contour lines instead of the
;        default filled contours.
;
;     MONOCHROME (boolean, input) -> Draw monochrome contour lines
;
;     BOXES (boolean, input) -> Draw filled polygons (boxes) for each
;        pixel. This keyword is overwritten by the line keyword.
;
;     IMAGE_STYLE (boolean, input) -> Draw data as interpolated color image.
;        This keyword is superseeded by line or boxes.
;
;     WINDOWSIZE (2-integer, input) -> Size of the drawing widget
;
;     CHARSIZE (float, input) -> The size of annotation characters and
;        axis labels.
;
;     VARIABLE (objectref, input) -> Instead of passing a data array,
;        you can also pass a mgs_(base)variable object reference.
;        *** Explain handling of dimensions ***
;        *** Currently, VARIABLE must be a 2-D variable !! ***
;
;     OBJECT (objectref, output) -> The MGS_Contour procedure operates
;        internally by defining and using an MGS_Contour object. If
;        you would like to keep the object at the end of the
;        procedure, pass a named variable via this keyword.
;
; ***** more keywords:
;    STYLE = 'work', 'presentation', 'publication'  sets various
;    parameters
;    CHARSIZE, FONT, ....
;    TITLE, XTITLE, YTITLE, ....
; ********************************
; MODIFICATION HISTORY:
;     mgs, 09 Oct 2000: VERSION 1.0
;-


PRO MGS_Contour, data, x, y, $
                 name=name,          $  ; The variable name
                 long_name=long_name, $ ; The variable long name
                 units=units, $         ; The variable unit
                 levels=levels,      $  ; The contour levels ...
                 nlevels=nlevels,    $  ; ... or the number of contour levels ...
                 minvalue=minvalue,  $  ; ... and the minimum data value ...
                 maxvalue=maxvalue,  $  ; ... and the maximum data value
                 log=log,            $  ; Use logarithmic spacing
                 color=color,        $  ; The drawing color for axes and lines
                 backcolor=backcolor,$  ; The background color
                 c_colors=c_colors,  $  ; The contour or fill colors
                 ctable=ctable, $ ; The color table to be used
                 map=map,            $  ; Draw data on map
                 limit=limit,        $  ; Range for X and Y (as in map_set)
                 grid_file=grid_file,$  ; Read lons and lats from (netcdf) file
                 lines=lines,        $  ; Contour lines instead of filled contours
                 monochrome=monochrome, $  ; Monochrome contours
                 boxes=boxes,        $     ; Filled polygons
                 image_style=image_style,  $ ; Interpolated image
                 windowsize=windowsize,  $ ; Size of drawing widget
                 charsize=charsize,  $     ; The character size
                 variable=theVariable,  $  ; A variable object in place of data, x, and y
                 debug=debug,     $
                 _Extra=extra,    $
                 object=theObject          ; The contour object



   ;; 1. Preparations:
   ;; Check Data, X, and Y, and create variable object(s)
   ;; (Dimensionality check is done within object)
   IF N_Elements(data) EQ 0 THEN BEGIN
      IF NOT Obj_Valid(theVariable) THEN BEGIN
         Message, 'You must pass either DATA or a VARIABLE object!'
         RETURN
      ENDIF
   ENDIF
   IF N_Elements(data) GT 1 THEN BEGIN
      thisVariable = Obj_New('MGS_Variable', data, name=name, $
                             long_name=long_name, unit=units)
   ENDIF ELSE IF Obj_Valid(theVariable) THEN BEGIN
      ;; create a local copy of the variable object
      thisVariable = theVariable->Copy()
   ENDIF ELSE BEGIN
      Message, 'DATA must be an array!'
      RETURN
   ENDELSE
   IF NOT Obj_Valid(thisVariable) THEN $
      Message, 'Variable object not valid! Please contact author!'


   thisVariable->GetProperty, dims=dims
print,'### dims(variable) = ',dims

   IF N_Elements(X) LE 1 THEN BEGIN
      IF N_Elements(dims) GT 0 THEN X = lindgen(dims[0])
   ENDIF
   IF N_Elements(Y) LE 1 THEN BEGIN
      IF N_Elements(dims) GT 1 THEN Y = lindgen(dims[1])
   ENDIF
   IF N_Elements(X) GT 0 THEN BEGIN
      thisX = Obj_New('MGS_Variable', X, name='X')
   ENDIF
   IF N_Elements(Y) GT 0 THEN BEGIN
      thisY = Obj_New('MGS_Variable', Y, name='Y')
   ENDIF

   ;; Read coordinates from file if requested
   IF N_Elements(grid_file) GT 0 THEN BEGIN
      ncdf_read, filename=grid_file[0], varnames=varnames
      ;; Look for longitude and latitude variables
      varnames = StrLowCase(varnames)
      wlon = Where(varnames EQ 'lon' OR varnames EQ 'longitude', success0)
      wlat = Where(varnames EQ 'lat' OR varnames EQ 'latitude', success1)
      IF (success0 GT 0 AND success1 GT 0) THEN BEGIN
         ncdf_read, grid, variables=[ varnames[wlon], varnames[wlat] ], /No_Dimensions
         ;; Convert them to X and Y variable objects unless these
         ;; were previously defined
         IF NOT Obj_Valid(thisX) THEN $
            thisX = Obj_New('MGS_Variable', grid.(0), $
                            name='longitude', long_name='Longitude', unit='degrees')
         IF NOT Obj_Valid(thisY) THEN $
            thisY = Obj_New('MGS_Variable', grid.(1), $
                            name='latitude', long_name='Latitude', unit='degrees')
      ENDIF
   ENDIF

print,'########'
;help,thisX,thisY
;thisX->GetData,XX
;help,XX
;thisY->GetData,YY
;help,YY
   ;; Determine if contour is on a map and rename coordinate variables
   ;; in this case
   ismap = Keyword_Set(map)
   IF ismap THEN BEGIN
      ;; Replace coordinate names with longitude and latitude
      IF Obj_Valid(thisX) THEN thisX->SetProperty, name='longitude', $
         long_name='Longitude', unit='degrees'
      IF Obj_Valid(thisY) THEN thisY->SetProperty, name='latitude', $
         long_name='Latitude', unit='degrees'
   ENDIF

   ;; Link coordinate variables to data variable
   linkX = 1
   linkY = 1
   IF Obj_Valid(theVariable) THEN BEGIN
      ;; Test if variable dimensions are valid object references
      ;; (this is limitation to 2D variable objects as argument)
      thisVariable->GetDimVar, 1, oref
      linkX = 1- Obj_Valid(oref)
      thisVariable->GetDimVar, 2, oref
      linkY = 1- Obj_Valid(oref)
   ENDIF
   IF linkX AND Obj_Valid(thisX) THEN thisVariable->SetDimVar, 1, thisX
   IF linkY AND Obj_Valid(thisY) THEN thisVariable->SetDimVar, 2, thisY

   ;; 2. Create the contour object
   theObject = Obj_New('MGS_Contour', thisVariable, $
                       levels=levels,  $
                       nlevels=nlevels, $
                       minvalue=minvalue,  $
                       maxvalue=maxvalue,  $
                       log=log,  $
                       color=color, $
                       drawcolor=color,    $
                       backcolor=backcolor,$
                       c_colors=c_colors,  $
                       ctable=ctable, $
                       ismap=ismap,        $
                       limit=limit,        $
                       lines=lines,        $
                       monochrome=monochrome,  $
                       boxes=boxes,        $
                       image_style=image_style,  $
                       windowsize=windowsize,  $
                       debug=debug, $
                       charsize=charsize, $
                       _Extra=extra)


   IF NOT Obj_Valid(theObject) THEN RETURN

   ;; 3. Display the contour plot
   theObject->Show


END
