;+
; NAME:
;    MGS_GUIDemoObject
;
; PURPOSE:
;    This object widget demonstrates the process of creating a widget
;  application with the MGS_BaseGUI class hierarchy. It builds a
;  widget containing three color pickers together with one text input
;  field. All of these are realized as compound widgets and managed
;  completely via the generic BaseGUI's compound container. Just see
;  how easy this is!
;
; CATEGORY:
;    Widget Objects
;
; CALLING SEQUENCE:
;    thedemo = Obj_New('MGS_GUIDemoObject' [,keywords])
;    thedemo->GUI, /block
;    state = thedemo->GetValue()
;    print,'Color values : ',state.colors, '  Text field: ',state.value
;    Obj_Destroy, thedemo
;
;    You can also use the example procedure at the end of this program
;    file: type ".run mgs_guidemoobject__define" then "example"
;
; ARGUMENTS:
;
; KEYWORDS:
;    color1 (string, input) -> The name of the first color
;    color2 (string, input) -> The name of the second color
;    color3 (string, input) -> The name of the third color
;    value  (any type, input) -> The initial value of the text input
;         field
;    
;    Several other keywords from MGS_Field and MGS_Drawcolor can be
;    passed, and some of these may actually work ;-)
;
;    inherited keywords:
;    widget_title (string, input) -> The title of the top level
;          base. This string is displayed as window title AND at the
;          top of the dialog window.
;
;    widget_defaultfont (string, input) -> A font name to be used for
;          the text fields in the widget. Note: Font names are
;          platform dependent!
;
;    widget_labelfont (string, input) -> A font name to be used for
;          the title labels of widget sections. Note: Font names are
;          platform dependent!
;
; MODIFICATION HISTORY:
;    mgs, 04 Apr 2001: Version 1.0 released
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################



; -----------------------------------------------------------------------------
; GetState:
;   This method returns the currently selected color names in a
; structure.

FUNCTION MGS_GUIDemoObject::GetState

   self->GetProperty, color1=color1, color2=color2, color3=color3, value=value

   IF N_Elements(color1) EQ 0 THEN print, 'Color 1 undefined'
   IF N_Elements(color2) EQ 0 THEN print, 'Color 2 undefined'
   IF N_Elements(color3) EQ 0 THEN print, 'Color 3 undefined'
   IF N_Elements(value) EQ 0 THEN print, 'Input field value undefined'

   RETURN, { colors: [color1, color2, color3], value:value }

END


; -----------------------------------------------------------------------------
; BuildGUI:  (private)
;   This method only adds a category label to the layout widget. All
; the compound widgets will be added automatically.

PRO MGS_GUIDemoObject::BuildGUI

   ;; All we need to do here is supply a title label
   labelID = Widget_Label(self.layoutID, Value='Color selector', $
                         font=self.labelfont)

END


; -----------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user. Normally, the user should use the GetState() method, or more
; specific, GetValue(), to retrieve the object state in a usable form.

PRO MGS_GUIDemoObject::GetProperty, $
   Color1=color1, $
   Color2=color2, $
   Color3=color3, $
   Value=value,   $
   _Ref_Extra=extra         ; Inherited and future keywords
                              ;
                              ; Inherited keywords:
                              ; name      : The variable name
                              ; uvalue    : a user-defined value
                              ; window_title
                              ; widget_title
                              ; widget_defaultfont
                              ; widget_labelfont
                              ; set_value_pro
                              ; get_value_func


   ;; Get properties from base object
   self->MGS_BaseGUI::GetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving object properties!'
      RETURN
   ENDIF

   IF Arg_Present(color1) GT 0 THEN BEGIN
      cobj = self.compound->Get(name='color1')
      IF Obj_Valid(cobj) THEN cobj->GetProperty, color=color1 ELSE color1='NONE'
   ENDIF 
   IF Arg_Present(color2) GT 0 THEN BEGIN
      cobj = self.compound->Get(name='color2')
      IF Obj_Valid(cobj) THEN cobj->GetProperty, color=color2 ELSE color2='NONE'
   ENDIF
   IF Arg_Present(color3) GT 0 THEN BEGIN
      cobj = self.compound->Get(name='color3')
      IF Obj_Valid(cobj) THEN cobj->GetProperty, color=color3 ELSE color3='NONE'
   ENDIF
   IF Arg_Present(value) GT 0 THEN BEGIN
      cobj = self.compound->Get(name='field1')
      IF Obj_Valid(cobj) THEN cobj->GetProperty, value=value ELSE value='NO_VALID_DATA'
   ENDIF 

END


; -----------------------------------------------------------------------------
; SetProperty:
; This method sets specific object values.
; NOTE: XSize and YSize only take effect when the widget is rebuilt.

PRO MGS_GUIDemoObject::SetProperty, $
   Color1=color1, $
   Color2=color2, $
   Color3=color3, $
   Value=value,   $
   _Extra=extra  ; Extra keywords from inherited objects
                                   ; Inherited keywords:
                                   ; name      : The variable name
                                   ; no_copy   : Don't keep local copy
                                   ;             of uvalue
                                   ; no_dialog : Don't display
                                   ;             interactive dialogs
                                   ; uvalue    : a user-defined value
                                   ; window_title
                                   ; widget_title
                                   ; row_layout
                                   ; no_frame
                                   ; widget_defaultfont
                                   ; widget_labelfont
                                   ; set_value_pro
                                   ; get_value_func


   ;; Set Properties of base object
   self->MGS_BaseGUI::SetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error setting object properties'
      RETURN
   ENDIF

   IF N_Elements(color1) GT 0 THEN BEGIN
      cobj = self.compound->Get(name='color1')
      IF Obj_Valid(cobj) THEN cobj->SetProperty, color=color1
   ENDIF
   IF N_Elements(color2) GT 0 THEN BEGIN
      cobj = self.compound->Get(name='color2')
      IF Obj_Valid(cobj) THEN cobj->SetProperty, color=color2
   ENDIF
   IF N_Elements(color3) GT 0 THEN BEGIN
      cobj = self.compound->Get(name='color3')
      IF Obj_Valid(cobj) THEN cobj->SetProperty, color=color3
   ENDIF
   IF N_Elements(value) GT 0 THEN BEGIN
      cobj = self.compound->Get(name='field1')
      IF Obj_Valid(cobj) THEN cobj->SetProperty, value=value
   ENDIF

   ;; Make sure object is up-to-date and redisplay
   self->UpdateObject
   self->Show

END

; -----------------------------------------------------------------------------
; Cleanup:
;   It is not necessary to overwrite the genric GUI object cleanup method!

; -----------------------------------------------------------------------------
; Init:
;   This method initializes the colorpicker object.

FUNCTION MGS_GUIDemoObject::Init, $
   Color1=color1, $
   Color2=color2, $
   Color3=color3, $
   Value=value,   $
   _Extra=extra  ; Extra keywords from inherited objects
                                ;
                                ; Inherited keywords (from
                                ; MGS_BaseGUI and MGS_BaseObject):
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value
                                ; window_title
                                ; widget_title
                                ; row_layout (ignored)
                                ; no_frame (ignored)
                                ; widget_defaultfont
                                ; widget_labelfont
                                ; set_value_pro
                                ; get_value_func


   ;; Initialize parent object
   IF not self->MGS_BaseGUI::Init(_Extra=extra) THEN RETURN, 0

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error initialising object'
      RETURN, 0
   ENDIF

   ;; Check keywords and parameters.
   colors = [ 'WHITE', 'BLACK', 'RED' ]
   IF N_Elements(color1) GT 0 THEN colors[0] = color1
   IF N_Elements(color2) GT 0 THEN colors[1] = color2
   IF N_Elements(color3) GT 0 THEN colors[2] = color3

   ;; Create compound widget objects (color pickers)
   FOR i=0,2 DO BEGIN
      label = 'Color '+StrTrim(i+1,2)
      colorObj = Obj_New('MGS_DrawColor', $
                         color=colors[i], $
                         index=!D.Table_Size-i-1, $
                         LabelText=label, $
                         LabelSize=76, $
                         /Align_Left, $
                         Window_Title=label, $
                         Name=StrLowCase(StrCompress(label,/remove_all)), $
                         _Extra=extra)
      IF Obj_Valid(colorObj) THEN self.compound->Add, colorObj
   ENDFOR

   ;; Create text input field
   label = 'Field 1'
   fieldObj = Obj_New('MGS_Field', $
                      value=value,  $
                      LabelText='', $
                      LabelSize=76, $
                      /Align_Left, $
                      Window_Title=label, $
                      Name=StrLowCase(StrCompress(label,/remove_all)), $
                      _Extra=extra)
   IF Obj_Valid(fieldObj) THEN self.compound->Add, fieldObj

   ;; Reset some of the generic default properties
   IF self.window_title EQ 'Generic widget object' THEN $
      self.window_title = 'Color Chooser'
   self.layout_frame = 0    ;; Don't put a frame around this widget

   RETURN, 1
END


; -----------------------------------------------------------------------------
; MGS_GUIDemoObject__Define:
; This is the object definition for the color picker object.
; It inherits from MGS_BaseGUI which provides the generic widget
; functionality, and from MGS_BaseObject the abilities to set and
; query an object name and a uvalue. The base object also provides a
; general method for display of error messages which can be directed
; to a message dialog or to the log screen via the no_dialog flag.

PRO MGS_GUIDemoObject__Define

   struct = { MGS_GUIDemoObject, $

              inherits MGS_BaseGUI  }

END


; -----------------------------------------------------------------------------
; Example:
;    Demonstrate the functionality of this object

PRO Example, block=block, object=object, _EXTRA=extra

   thegui = Obj_New('MGS_GUIDemoObject', _Extra=extra)
   thegui->GUI, block=keyword_Set(block)
   state = thegui->GetState()
   print, 'Selected colors : ',state.colors,'  Value of input field: ',state.value
   IF Arg_Present(object) EQ 0 THEN Obj_Destroy, thegui ELSE object=thegui

END
