;+
; NAME:
;   MGS_VarContainer (Object)
;
; PURPOSE:
;      This object provides a convenient and powerful way to store a
;   set of variables (aka dataset) and exploit their common
;   properties.  
;      This object extends the MGS_Container object ...
;
; AUTHOR:
;   Dr. Martin Schultz
;   Max-Planck-Institut fuer Meteorologie
;   Bundesstr. 55, D-20146 Hamburg
;   email: martin.schultz@dkrz.de
;
; CATEGORY:
;   Objects, General Programming
;
; CALLING SEQUENCE:
;
; KEYWORDS TO THE INITIALIZATION ROUTINE:
;   None.
;
; PROCEDURES AND METHODS:
;
; NON-OBJECT PROCEDURES:
;   None
;   
; PROCEDURE METHODS:
;
; FUNCTION METHODS:
;
; MODIFICATION HISTORY:
;
;   mgs, 16 Mar 2001 : VERSION 1.00
;
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2000-2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution. 
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################



;---------------------------------------------------------------------------
; MakeVarSelList (private):
;   This method compiles basic information about all or selected
; variables and returns this as a string array that can be used
; e.g. to build a droplist widget. If no variables are found or an
; error occurs, the result will be an empty string. 
;
; Note: unlike the Get method which defaults to return the first list
; element, this method returns all variables by default, so the All
; keyword is not needed. Extra keywords are passed to the Get method.

FUNCTION MGS_VarContainer::MakeVarSelList, $
                         Position=position, $
                         Name=name, $
                         _Extra=extra
   
   
   get_all = (N_Elements(position) + N_Elements(name)) EQ 0

   thevars = self->Get(Position=position, name=name, all=get_all, $
                       _Extra=extra)

   retval = StrArr(N_Elements(thevars))

   ;; loop through variables and get their names and the dimension
   ;; names
   FOR i = 0L, N_Elements(thevars)-1 DO BEGIN
      thevars[i]->GetProperty, name=name, dimnames=dimnames, dims=dims
      thisstring = name + '('
      FOR j = 0, (N_Elements(dimnames) < N_Elements(dims))-1 DO BEGIN
         IF dimnames[j] EQ '' THEN BEGIN
            thisstring = thisstring + StrTrim(dims[j],2) + ','
         ENDIF ELSE BEGIN
            thisstring = thisstring + dimnames[j] + ','
         ENDELSE 
      ENDFOR
      len = StrLen(thisstring)
      IF StrMid(thisstring, len-1, 1) EQ ',' THEN $
         thisstring = StrMid(thisstring,0,len-1) + ')'  $
      ELSE  $
         thisstring = thisstring + ')'
      retval[i] = thisstring
   ENDFOR

   RETURN, retval
END


;---------------------------------------------------------------------------
; Get:
; This method returns object references for objects that match
; specified criteria - or all objects if the All keyword is set.
; The IDL_VarContainer method Get is extended to include a Name
; keyword. All and Position take precedence over name. As in the
; original, count can be used to return the number of objects
; retrieved. 
; mgs, 05 Sep 2000: now returns object references in requested order
; unless sorted keyword is set. Note that this is counterintuitive
; because the inherited Get method sorts by default, so we have to
; "unsort". 

FUNCTION MGS_VarContainer::Get, All=all,   $
                      Position=position, $
                      Name=name,  $
                      Case_Sensitive=case_sensitive,  $
                      NoEmpty=NoEmpty,  $
                      Count=count,  $
                      Sorted=sorted,  $
                      _Ref_Extra=e


   ;; Initialize count value
   count = 0L
return, self->MGS_Container::Get(all=all,position=position,name=name)


   ;; If All or Position keywords are used, call inherited method
   IF Keyword_Set(all) THEN  $
     RETURN, self->IDL_Container::Get(/All, count=count)
   
   IF N_Elements(Position) GT 0 THEN BEGIN
      result = self->IDL_Container::Get(Position=position, count=count)
      IF Keyword_Set(sorted) EQ 0 AND N_Elements(position) GT 1 THEN $
         result = result[sort(position)]
      RETURN, result
   ENDIF

   ;; Look for named objects ?
   IF N_Elements(name) GT 0 THEN BEGIN
      ;; Find object positions
      namepos = self->GetPosition(name, $
                                  case_sensitive=case_sensitive,  $
                                  noempty=noempty)

      ;; If no match, return empty object reference
      IF namepos[0] LT 0 THEN BEGIN
         dummy = obj_new()
         RETURN, dummy
      ENDIF
      ;; Return objects
      result = self->IDL_Container::Get(Position=namepos, count=count)
      IF Keyword_Set(sorted) EQ 0 AND N_Elements(namepos) GT 1 THEN  $
         result = result[sort(namepos)]
      RETURN, result
   ENDIF

   ;; Otherwise call inherited method with extra keywords
   RETURN, self->IDL_Container::Get(_extra=e, count=count)

END


;---------------------------------------------------------------------------
; IsContained:
; This method tests if a specific object reference or an object with 
; a certain name is in the container. The IsContained method of
; IDL_Container is extended so that it checks the type of objref.
; If ChildObject is a string or string array, a pattern match is
; performed. For IDL >= 5.3 the StrMatch function is used which allows
; Unix like wildcard search ('*', '?', and '[...]'). For IDL <= 5.2 a
; primitive wildcard search is implemented which only allows for a
; trailing '*'.


FUNCTION MGS_CONTAINER::IsContained, ChildObject,   $
                      Position = Position,  $
                      Case_Sensitive=case_sensitive, $
                      NoEmpty=NoEmpty

   ;; If ChildObject is empty then return immediately
   IF N_Elements(ChildObject) EQ 0 THEN RETURN,0

   type = Size(ChildObject, /TName)

   CASE Type OF
      'OBJREF':  BEGIN
         result =  Self->IDL_Container::IsContained(ChildObject, Position = Position)
         RETURN, result
      END
      
      'STRING':  BEGIN        
         ;; For strings, interpret them as object names. Get the names of
         ;; all objects and apply (version dependent) pattern
         ;; matching.
         result = 0
         position = -1L
         names = self->GetNames()
         names = StrTrim( names, 2)
         IF NOT Keyword_Set(case_sensitive) THEN $
            names = StrUpCase(names)

         ;; Return if container has no objects stored
         IF self->Count() EQ 0 THEN RETURN, result

         ;; Loop through objlist and find matching indices
         FOR i=0L, N_Elements(ChildObject)-1 DO BEGIN
            testname = StrTrim( ChildObject[i], 2)

            IF (!Version.Release GE 5.3) THEN BEGIN
               ;; Use the StrMatch function
               ;; StrMatch returns a logical array with 1 for every
               ;; string that matches the search expression
               lw = StrMatch(names,testname,fold_case=1-Keyword_Set(case_sensitive))
               ;; Convert logical array to index array
               w = Where(lw, wcnt)
            ENDIF ELSE BEGIN
               ;; Restricted wildcard use (only '*' at the end of the expression)
               ;; Check if wildcard is used in name
               IF NOT Keyword_Set(case_sensitive) THEN $
                  testname = StrUpCase(testname)
               sloppy = 0
               IF (( p = StrPos(testname,'*') )) GE 0 THEN BEGIN
                  l = StrLen(testname)
                  IF l-p NE 1 THEN BEGIN ; wildcard must be last character
                     Message, 'Invalid name specification '+ChildObject[i]+'!', /Continue
                     GOTO, skip_it
                  ENDIF
                  testname = StrMid(testname, 0, l-1)
                  sloppy = 1
               ENDIF

               ;; Look for matching object names
               IF sloppy THEN BEGIN
                  w = Where(StrPos(names,testname) EQ 0, wcnt)
               ENDIF ELSE BEGIN
                  w = Where(names EQ testname, wcnt)
               ENDELSE
            ENDELSE

            ;; Add indices to list
            IF wcnt GT 0 THEN position = [ position, w ]

skip_it:
         ENDFOR

         ;; Remove first dummy position if any names have been found
         ;; and set result value to 1
         IF N_Elements(position) GT 1 THEN BEGIN
            result = 1
            position = position[1:*]
         ENDIF

         ;; Remove position of unnamed objects if NoEmpty keyword is
         ;; set
         IF Keyword_Set(NoEmpty) AND position[0] GE 0 THEN BEGIN
            we = Where(names EQ '', wcnt)
            IF wcnt GT 0 THEN BEGIN
               ;; Convert index array to logical array
               ;; and delete "empty" elements
               larr = intarr(N_Elements(names))
               larr[position] = 1
               larr[we] = 0
               position = Where(larr, wcnt)
               IF wcnt EQ 0 THEN result = 0
            ENDIF
         ENDIF

         RETURN, result
      END

      ELSE:      BEGIN
         self->ErrorMessage, 'ChildObject must be of type OBJREF or STRING!'
         RETURN,0
      END
   ENDCASE
END 


;---------------------------------------------------------------------------
; Cleanup:
; This method frees all object values and destroys all objects stored
; in the container. This is already handled in IDL_Container, so
; Cleanup only needs to call the inherited object's method.

PRO MGS_VarContainer::Cleanup

   IF Ptr_Valid(self.global_attributes) THEN Ptr_Free, self.global_attributes

   self->MGS_Container::Cleanup

END


;---------------------------------------------------------------------------
; Init:
; This method initializes the object values. If a variable object
; reference (or a list of such references) is passed, these variables
; will be added to the container upon initialization. 

FUNCTION MGS_VarContainer::INIT,  $
                         variables=variables,  $
                         global_attributes=global_attributes, $
                         _Extra=extra

   RETURN, self->MGS_Container::Init()

   ;; Error handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error initializing variable container!'
      RETURN, 0
   ENDIF

   ;; Initialize global attributes pointer
   IF N_Elements(global_attributes) EQ 0 THEN BEGIN
      self.global_attributes = Ptr_New()
   ENDIF ELSE BEGIN
      self.global_attributes = Ptr_New(global_attributes)
   ENDELSE

   ;; Add variables if passed
   IF N_Elements(variables) GT 0 THEN BEGIN
      FOR i = 0L, N_Elements(variables)-1 DO BEGIN
         self->Add, variables[i]
         IF self.debug GT 0 THEN BEGIN
            variables[i]->GetProperty, name=name
            print,'  Added variable '+name
         ENDIF
      ENDFOR
   ENDIF

   RETURN, 1

END


;---------------------------------------------------------------------------
; MGS_VarContainer__Define:
; This is the object definition for the container object. It inherits
; everything from MGS_Container. 

PRO MGS_VarContainer__Define

   object_class = {  MGS_VarContainer,    $
                     global_attributes : Ptr_New(),  $

                     INHERITS  MGS_Container    $
                  }

END







