; method to transform a HANK netCDF file into a GEO netCDF file
PRO hank_ncfile::convert, ncfile_out, $
                         latmin, latmax, latdel, $
                         lonmin, lonmax, londel

     ; build lat/lon grid
     self->grid, latmin, latmax, latdel, lonmin, lonmax, londel

     ; OPEN output file (override existing one)
     fid = NCDF_CREATE(ncfile_out,/CLOBBER)

     ; DEFINITION mode
     NCDF_CONTROL, fid, /NOFILL
     
     ; dimensions
     for did=0,self.ndims-1 do begin
         CASE did OF
              ; substitute x --> longitude
              self.ix: lonid = NCDF_DIMDEF(fid, 'lon', n_elements(*self.lons))
              ; substitute y --> latitudes
              self.iy: latid = NCDF_DIMDEF(fid, 'lat', n_elements(*self.lats))
              ; time must be defined as unlimited
              self.itime: same_did = NCDF_DIMDEF(fid, 'time', /UNLIMITED )
              ; transfer all other dimensions input->output
              ELSE: same_did = NCDF_DIMDEF(fid, (*self.dnames)[did], (*self.dsizes)[did] )
         ENDCASE
     endfor


     ; latitude and longitude
     latvid = NCDF_VARDEF(fid, 'lat', [latid], /float) ; latvid=0
     lonvid = NCDF_VARDEF(fid, 'lon', [lonid], /float) ; lonvid=1
     if (latvid ne 0 or lonvid ne 1) then stop

     ; variables
     self.vidsout = ptr_new(lonarr(self.nvars)) 
     (*self.vidsout)[*] = -1 ; variable NOT written to output
     for vid=0, self.nvars-1 do begin ; all other variables

         ; determine correct datatype
         byte_type=0 & char_type=0 & double_type=0 & float_type=0 & long_type=0 & short_type=0
         inq = NCDF_VARINQ(self.fid,vid)
         CASE strlowcase(inq.datatype) OF
                 'byte':   byte_type=1
                 'char':   char_type=1
                 'double': double_type=1
                 'float':  float_type=1
                 'long':   long_type=1
                 'short':  short_type=1
         ENDCASE

         ; scalar variable
         if ((*self.vndims)[vid] eq 0) then begin
            (*self.vidsout)[vid] = NCDF_VARDEF(fid, (*self.vnames)[vid], $
                                               byte=byte_type, char=char_type, double=double_type, $
                                               float=float_type, long=long_type, short=short_type)

         ; non-scalar variable
         endif else begin

            ; possibly exclude variables 'lat' and 'lon', if present
            if (strlowcase((*self.vnames)[vid]) ne 'lat' $
                and strlowcase((*self.vnames)[vid]) ne 'lon') then begin

               ; must create array containing new dimensions dependence
               vdims = lonarr((*self.vndims)[vid]) 
               for idim = 0, (*self.vndims)[vid]-1 do begin
                   did = self->get_dimid_of_var(idim,vid)
                        if (did eq self.ix) then vdims[idim] = lonid $ ; 'x' --> 'lon'
                   else if (did eq self.iy) then vdims[idim] = latid $ ; 'y' --> 'lat'
                   else                          vdims[idim] = did
               endfor
               (*self.vidsout)[vid] = NCDF_VARDEF(fid, (*self.vnames)[vid], vdims, $
                                                  byte=byte_type, char=char_type, double=double_type, $
                                                  float=float_type, long=long_type, short=short_type)

            endif ; not lat or lon

         endelse ; non-scalar variable

         ; variable written to output -> transfer attributes
         if ((*self.vidsout)[vid] ge 0) then begin
               result = NCDF_VARINQ(self.fid,vid)
               for iatt=0, result.natts-1 do begin
                   attname = NCDF_ATTNAME(self.fid,vid,iatt)
                   r = NCDF_ATTCOPY(self.fid, vid, attname, fid, (*self.vidsout)[vid])
               endfor
               ; add missing value for non-scalar variables
               if ((*self.vndims)[vid] gt 0) then $
                   NCDF_ATTPUT, fid, (*self.vidsout)[vid], 'missing_value', self.missing, /FLOAT
         endif

     endfor

     ; global attributes
     for ig=0,self.ngatts-1 do begin
         attname = ncdf_attname(self.fid, /global, ig)
         r = NCDF_ATTCOPY(self.fid, /IN_GLOBAL, attname, fid, /OUT_GLOBAL)
     endfor
     ; add base model
     NCDF_ATTPUT, fid, /GLOBAL, 'base model', 'HANK', /char

     ; OUTPUT mode
     NCDF_CONTROL, fid, /ENDEF

     ; write latitude, longitude to output
     NCDF_VARPUT, fid, latvid, *self.lats
     NCDF_VARPUT, fid, lonvid, *self.lons
     
     ; loop over input variables and write to output
     for vid=0, self.nvars-1 do begin
      if ((*self.vidsout)[vid] ge 0) then begin ; variable to be written out

         print,'processing variable:',(*self.vnames)[vid]

         ; dimension id --> variable dependence array
         dim2var = intarr(self.ndims) & dim2var[*] = -1
         for idim=0,(*self.vndims)[vid]-1 do begin
             did = self->get_dimid_of_var(idim,vid)
             dim2var[did] = idim
         endfor

         ; variable depends on x and y
         if (dim2var[self.ix] ge 0 and dim2var[self.iy] ge 0) then begin

             countin = intarr((*self.vndims)[vid])
             countin[ dim2var[self.ix] ] = (*self.dsizes)[self.ix]
             countin[ dim2var[self.iy] ] = (*self.dsizes)[self.iy]

             offsetin= intarr((*self.vndims)[vid])
             offsetin[ dim2var[self.ix] ] = 0
             offsetin[ dim2var[self.iy] ] = 0

             countout = intarr((*self.vndims)[vid])
             countout[ dim2var[self.ix] ] = n_elements(*self.lons)  ; 'lon' dimension replaced 'x'
             countout[ dim2var[self.iy] ] = n_elements(*self.lats)  ; 'lat' dimension replaced 'y'

             offsetout= intarr((*self.vndims)[vid])
             offsetout[ dim2var[self.ix] ] = 0
             offsetout[ dim2var[self.iy] ] = 0

             ; variable also depends on lev and time
             if (dim2var[self.ilev] ge 0 and dim2var[self.itime] ge 0) then begin

                 for itime=0, (*self.dsizes)[self.itime]-1 do begin 
                     countin[ dim2var[self.itime] ] = 1
                     offsetin[ dim2var[self.itime] ] = itime
                     countout[ dim2var[self.itime] ] = 1
                     offsetout[ dim2var[self.itime] ] = itime

                     for ilev=0, (*self.dsizes)[self.ilev]-1 do begin
                         countin[ dim2var[self.ilev] ] = 1
                         offsetin[ dim2var[self.ilev] ] = ilev
                         countout[ dim2var[self.ilev] ] = 1
                         offsetout[ dim2var[self.ilev] ] = ilev

                         self->getput, vid,countin,offsetin, fid,(*self.vidsout)[vid],countout,offsetout

                     endfor
                 endfor

             ; variable depends on lev, not time
             endif else if (dim2var[self.ilev] ge 0) then begin

                for ilev=0, (*self.dsizes)[self.ilev]-1 do begin
                    countin[ dim2var[self.ilev] ] = 1
                    offsetin[ dim2var[self.ilev] ] = ilev
                    countout[ dim2var[self.ilev] ] = 1
                    offsetout[ dim2var[self.ilev] ] = ilev

                    self->getput, vid,countin,offsetin, fid,(*self.vidsout)[vid],countout,offsetout

                endfor

             ; variable depends on time, not on lev
             endif else if (dim2var[self.itime] ge 0) then begin

                for itime=0, (*self.dsizes)[self.itime]-1 do begin
                    countin[ dim2var[self.itime] ] = 1
                    offsetin[ dim2var[self.itime] ] = itime
                    countout[ dim2var[self.itime] ] = 1
                    offsetout[ dim2var[self.itime] ] = itime

                    self->getput, vid,countin,offsetin, fid,(*self.vidsout)[vid],countout,offsetout

                endfor

             ; variable just depends on x, y
             endif else begin
                   self->getput, vid,countin,offsetin, fid,(*self.vidsout)[vid],countout,offsetout
             endelse

         ; variable does not depend on x, y
         endif else begin
               ; read from input
               NCDF_VARGET, self.fid, vid, data
               ; write to output
               NCDF_VARPUT, fid, (*self.vidsout)[vid], data
         endelse

       endif ; variable to be written out
     endfor  ; loop over variables

     ; CLOSE filee
     NCDF_CLOSE, fid

END
