;+
; NAME:
;    mgs_contour (object)
;
; PURPOSE:
;    This is a
;
; CATEGORY:
;    (Objects)
;
; AUTHOR:
;    Martin G. Schultz
;    Max-Planck-Institut fuer Meteorologie
;    Bundesstr. 55
;    20146 Hamburg
;    email: martin.schultz@dkrz.de
;
; REQUIREMENTS:
;    Inherits from MGS_BaseObject (error handling, uvalue)
;    Uses David Fanning's GetColor, XColor, TVRead, FSC_PSConfig programs
;
; MODIFICATION HISTORY:
;    mgs, dd MM yyyy: VERSION 1.0
;-
;
; ***** TO DO NEXT:
;    - improve handling of colortables (seperate method for loadct,
;      reserve a few drawing colors - at least black and white)
;    - improve update of color table if number of levels change
;    - add "XColors" menu button and widget to select drawing colors
;    - get variable list object ready and allow interactive variable
;      selection and cross sections
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2000 Martin Schultz
; Some routines were copied and modified from David Fanning's zoombox
; program (see comments at individual methods or routines)
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################



; =============================================================================
; Non-method procedures (widget event handling)
; =============================================================================

; -----------------------------------------------------------------------------
; Rubberbox:
; This function generates a pointer to a rubberbox structure which
; should be included in the info structure to be stored at the top
; level base if a draw widget shall allow for zooming.
; Everything else is handled by the DrawRubberbox procedure.

FUNCTION Rubberbox, xsize, ysize, color=color, linestyle=linestyle

   ;; Create the structure and 4 pixmap windows to store the original
   ;; window content below the rubberbox lines

   curwin = !D.Window

   Window, /free, xsize=xsize, ysize=2, /pixmap
   xbuf0 = !D.Window
   Window, /free, xsize=xsize, ysize=2, /pixmap
   xbuf1 = !D.Window
   Window, /free, xsize=2, ysize=ysize, /pixmap
   ybuf0 = !D.Window
   Window, /free, xsize=2, ysize=ysize, /pixmap
   ybuf1 = !D.Window

   WSet, curwin
;; print,'## RUBBERBOX SET UP. Curwin = ',curwin
   IF N_Elements(color) GT 0 THEN $
      thiscolor = long(color[0])  $
   ELSE  $
      thiscolor = long(!P.color)

   IF N_Elements(linestyle) GT 0 THEN $
      thislinestyle = linestyle[0]  $
   ELSE  $
      thislinestyle = 0

   result = { xsize:long(xsize),   $
              ysize:long(ysize),   $
              x0:0L,  x1:0L,  y0:0L,  y1:0L,  $
              xbuf0:xbuf0,  xbuf1:xbuf1,      $
              ybuf0:ybuf0,  ybuf1:ybuf1,      $
              initialized:0,                  $
              color:thiscolor,                $
              linestyle:thislinestyle }

   RETURN, Ptr_New(result)
END


; -----------------------------------------------------------------------------
; DrawRubberbox:
; This procedure draws a rubberbox rectangle. If the Initialize
; keyword is set, the two coordinate values are interpreted as the
; fixed corner coordinates and no restoring takes place. Otherwise,
; the old window content is restored before the new moving corner
; coordinates are saved and the new rectangle is drawn. If the Delete
; keyword is set, the window content will be restored and the pixmap
; windows will be freed (With the delete keyword, x, and y need not be
; given).
; The Rubberbox argument is a pointer as returned by the Rubberbox
; procedure.

PRO DrawRubberbox, rubberbox, x, y, Initialize=Initialize, Delete=Delete

   ;; Save current window index
   curwin = !D.Window

   ;; Shortcut
   r = *rubberbox

   ;; Restore old window content unless Initialize keyword is set
   IF Keyword_Set(Initialize) EQ 0 THEN BEGIN
      IF r.initialized EQ 0 THEN $
         Message, 'DrawRubberbox: asked to restore window content uninitialized!'
      Device, Copy=[0,0,r.xsize,2,0,r.y0,r.xbuf0]
      Device, Copy=[0,0,r.xsize,2,0,r.y1,r.xbuf1]
      Device, Copy=[0,0,2,r.ysize,r.x0,0,r.ybuf0]
      Device, Copy=[0,0,2,r.ysize,r.x1,0,r.ybuf1]
   ENDIF

   ;; Delete rubberbox if requested
   IF Keyword_Set(Delete) THEN BEGIN
      WDelete, r.xbuf0
      WDelete, r.xbuf1
      WDelete, r.ybuf0
      WDelete, r.ybuf1
      r.xsize = -1
      r.ysize = -1
      r.x0 = -1
      r.y0 = -1
      r.initialized = 0
;;      print,'Rubberbox deleted'
      *rubberbox = r
      RETURN
   ENDIF

   ;; Set new coordinates (fixed coordinates if Initialize is set)
   ;; Initialize also copies the two lines crossing x0 and y0
   IF Keyword_Set(Initialize) THEN BEGIN
      r.x0 = x
      r.y0 = y
      WSet, r.xbuf0
      Device, Copy=[0,y,r.xsize,2,0,0,curwin]
      WSet, r.ybuf0
      Device, Copy=[x,0,2,r.ysize,0,0,curwin]
      r.initialized = 1
;;      print,'Rubberbox initialized at x,y=',x,y
   ENDIF

   r.x1 = x
   r.y1 = y

   ;; Copy lines through new rectangle corners into buffer
;;   print,'Rubberbox: curwin=',curwin,'   x,y=',x,y
   WSet, r.xbuf1
   Device, Copy=[0,y,r.xsize,2,0,0,curwin]
   WSet, r.ybuf1
   Device, Copy=[x,0,2,r.ysize,0,0,curwin]

   ;; Reset window to current
   WSet, curwin

   ;; Draw new rectangle (if extension ne 0)
   rectangle, [ r.x0, r.y0, r.x1, r.y1], px, py
   PlotS, px, py, /DEVICE, color=r.color, line=r.linestyle
   PlotS, px+1, py+1, /DEVICE, color=r.color, line=r.linestyle

   *rubberbox = r
END


; -----------------------------------------------------------------------------
; MGS_Contour_EventHandler:
; This is the main event handler for the MGS_Contour object. It
; dispatches the event to the appropriate object method.

PRO MGS_Contour_EventHandler, event

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      IF Obj_Valid(info.object) THEN BEGIN
         info.object->ErrorMessage, ['Error in event handler!', $
                                     'Event : '+String(tag_names(event))]
      ENDIF ELSE BEGIN
         Message, 'Error in event handler!'+  $
            'Event : '+String(event), /Continue
      ENDELSE
      RETURN
   ENDIF

   thisEvent = Tag_Names(event, /Structure_Name)

   CASE thisEvent OF

;      'FSC_DROPLIST_EVENT' : BEGIN
;         Widget_Control, event.id, Get_UValue=droplistObj
;         instructions = droplistObj->GetUValue()
;      END

;      'FSC_FIELD': BEGIN
;         Widget_Control, event.id, Get_UValue=instructions
;      END

      'WIDGET_BUTTON': BEGIN
         Widget_Control, event.id, Get_UValue=info
      END

      'WIDGET_DRAW': BEGIN
         print,'WIDGET_DRAW event !!'
;         Widget_Control, event.id, Get_UValue=info
      END

      ELSE: BEGIN
         print,'### UNKNOWN event !!'
      END

   ENDCASE

   ;; Call the event handling method of the object that manages the
   ;; event generating widget
   IF info.method NE 'NULL' THEN BEGIN
      IF ChkStru(info,'ARGUMENT') THEN BEGIN
         Call_Method, info.method, info.object, info.argument, event
      ENDIF ELSE BEGIN
         Call_Method, info.method, info.object, event
      ENDELSE
   ENDIF

END


; -----------------------------------------------------------------------------
; EventInfo:
;   This procedure prints the information about an IDL mouse event in a
; condensed form.

PRO EventInfo, event

    thisEvent = Tag_Names(event, /Structure_Name)
    IF ChkStru(event,['x','y','press','release','clicks']) THEN BEGIN
       possibleEventTypes = [ 'DOWN', 'UP', 'MOTION', 'SCROLL', 'EXPOSE', 'UNKNOWN' ]
       thisEvent = possibleEventTypes(event.type)
       print,'### Event: ',thisevent,' ID=',StrTrim(event.id,2),  $
            ' x,y=',StrTrim(event.x,2),',', $
            StrTrim(event.y,2),' press, release, clicks=',  $
            StrTrim(fix(event.press),2),',', $
            StrTrim(fix(event.release),2),',',StrTrim(event.clicks,2),  $
            ' Caller=',Routine_Name(/Caller)
    ENDIF ELSE BEGIN
       help,event,/Structure
    ENDELSE
END


; -----------------------------------------------------------------------------
; MGS_Contour_ButtonEvent:
; This is the event handler for button events in the draw widget.
; This routine is based on David Fanning's ZOOMBOX program
; (version of 24 April 2000).
; This event handler ONLY responds to button down and expose events from the
; draw widget. If it gets a DOWN event, it does three things: (1) sets
; the static corners of the box, (2) changes the event handler for
; the draw widget to MGS_Contour_DRAWZOOMBOX and turns on MOTION events,
; and (3) sets up the polygon object that is the box.

PRO MGS_Contour_ButtonEvent, event

   possibleEventTypes = [ 'DOWN', 'UP', 'MOTION', 'SCROLL', 'EXPOSE', 'UNKNOWN' ]
   thisEvent = possibleEventTypes(event.type)
   pressed_left   = (event.press AND 1) GT 0
   pressed_middle = (event.press AND 2) GT 0
   pressed_right  = (event.press AND 4) GT 0

   ;; EMulate 3-button mouse
   IF pressed_middle EQ 0 THEN $
      pressed_middle = ((event.press AND 4) GT 0) AND ((event.press AND 1) GT 0)
   IF pressed_middle THEN BEGIN
      pressed_left = 0
      pressed_right = 0
   ENDIF

   ;; Get info structure
   Widget_Control, event.top, Get_UValue=info, /No_Copy

   ;; DIsplay event information if requested by object
   IF Obj_Valid(info.object) THEN BEGIN
      info.object->GetProperty, report_events=report_events
      IF report_events THEN EventInfo, event
   ENDIF

   ;; Process event depending on its type
   CASE thisEvent OF

      'EXPOSE': BEGIN           ; Redraw the view.
         print,'MGS_Contour_ButtonEvent: ### This event should not occur!'
         Widget_Control, event.top, Set_UValue=info, /No_Copy
         RETURN
         ;; Widget_Control, event.top, Get_UValue=info
         ;; IF NOT Obj_Valid(info.object) THEN $
         ;;    Message, 'Who killed the object ?'
         ;; WSet, info.object->GetWID()
         ;; info.object->Show
         ;; for object graphics see David Fanning's zoombox program
      END

      'DOWN': BEGIN
         IF NOT Obj_Valid(info.object) THEN $
            Message, 'Who killed the object ?'

         ;; Action depends on whether left or middle mouse button was
         ;; pressed
         IF pressed_left THEN BEGIN

            IF event.clicks GT 1 THEN BEGIN
               ;; This event is a double click event. Note that the
               ;; event handler has not been changed in this case!
               ;; Later, this should be used to show a profile
               ;; perpendicular to the current view plane. For now, just print
               ;; the value of the underlying point.

               ;; Convert device to data coordinates and find nearest data value
               dc = Convert_Coord(event.x, event.y, /Device, /To_Data)
               value = info.object->GetNearestData( dc[0,0], dc[1,0] )
               print, 'Nearest data value = ',value

               ;; Put the info structure back into its storage location.
               Widget_Control, event.top, Set_UValue=info, /No_Copy
               RETURN
            ENDIF

            ;; Set the static corners of the box to current
            ;; cursor location.
            xs = event.x
            ys = event.y

            ;; Turn MOTION events ON.
            Widget_Control, event.id, Draw_Motion_Events=1

            ;; Start rubberbox for zoom in
            IF Ptr_Valid(info.rubberbox) THEN Message,'Rubberbox pointer already valid!'

            ;; Initialize the polygon
            info.object->GetProperty, xsize=xsize, ysize=ysize,  $
               rubberboxcolor=rboxcol
            rubberbox = Rubberbox(xsize, ysize, line=4, color=rboxcol)
            DrawRubberbox, rubberbox, xs, ys, /Initialize
            info.rubberbox = rubberbox

         ENDIF ELSE IF pressed_middle THEN BEGIN
            ;; Zoom out
            info.object->SetProperty, limit=0
            info.object->Show

;;            print,'### middle button pressed: turn off motion events and delete rubberbox'
            ;; Turn motion events off
            Widget_Control, event.id, Draw_Motion_Events=0

            ;; Delete rubberbox
            Ptr_Free, info.rubberbox
         ENDIF

      END

      'UP': BEGIN
         ;; Up events should only occur after DOWN events, i.e. MOTION events
         ;; have been turned on and the rubberbox is initialized *OR* the middle
         ;; mouse button was pressed.
         ;; You need to erase the zoombox, turn motion events
         ;; OFF, and draw the "zoomed" plot.

         ;; Turn motion events off
         Widget_Control, event.id, Draw_Motion_Events=0

         IF NOT Ptr_Valid(info.rubberbox) THEN BEGIN
;;            print,'%%% Up event: Rubberbox pointer void!'
            Widget_Control, event.top, Set_UValue=info
            RETURN
         ENDIF

         ;; Erase the zoom box
         x0 = (*info.rubberbox).x0
         y0 = (*info.rubberbox).y0
         ;; DrawRubberbox, info.rubberbox, /Delete

         ;; ... or actually remove it from the info structure ???
         Ptr_Free, info.rubberbox

         ;; Draw the "zoomed" plot.
         ;; Get the DATA coordinates for LIMIT
         ;; ####
         ;; window size
         IF NOT Obj_Valid(info.object) THEN $
            Message, 'Who killed the object ?'
         info.object->GetProperty, xsize=xsize, ysize=ysize
         event.x = 0 > event.x < (xsize-1)
         event.y = 0 > event.y < (ysize-1)
         x = [x0, event.x]
         y = [y0, event.y]

         ;; If the user just clicked in the window, return with no
         ;; further action. Allow for 1 pixel tolerance.
         IF ABS(x0-event.x) LE 1   $
            OR ABS(y0-event.y) LE 1 THEN BEGIN
            Widget_Control, event.top, Set_UValue=info, /No_Copy
            RETURN
         ENDIF

         ;; Make sure the x and y values are ordered as [min, max].
         IF x[0] GT x[1] THEN x = [ x[1], x[0] ]
         IF y[0] GT y[1] THEN y = [ y[1], y[0] ]

         ;; Convert device to data coordinates and set limit property
         ;; Note: limit stored as y0, x0, y1, x1

         ;; Make sure coordinate conversion is ok for map
         ;; #### THIS IS NOT SATISFACTORY YET !! ####
         WSet, info.object->GetWID()
;; print,'before : X=',x,'Y=',y,'####----####'
         boundary = Round(convert_coord(!x.window,!y.window,/Normal,/To_Device))
         ;; #### How do I find out if map is active? And how do I find
         ;; out the first and last valid pixel of a map?
         ;; For maps, the
         ;; following still produces NaN in coordinate conversion:
         ;; x = ( x > boundary[0,0] ) < boundary[0,1]
         ;; y = ( y > boundary[1,0] ) < boundary[1,1]
         ;; #### Therefore be somewhat "generous" here:
         x = ( x > (boundary[0,0]+6) ) < (boundary[0,1]-6)
         y = ( y > (boundary[1,0]+6) ) < (boundary[1,1]-6)

;; print,'!X.Window, !Y.Window = ',!X.Window, !Y.Window
;; print,'BOUNDARY = ',boundary,'after  X=',x,'Y=',y
         dc = Convert_Coord(x, y, /Device, /To_Data)
;; print,'### dc->limit = ',[ dc[1,0], dc[0,0], dc[1,1], dc[0,1] ]
         info.object->SetProperty,limit = [ dc[1,0], dc[0,0], dc[1,1], dc[0,1] ]

         ;; Show updated image
         info.object->Show

      END

      'MOTION': BEGIN
         ;; Motion events should only occur after DOWN events, i.e. MOTION events
         ;; have been turned on and the rubberbox is initialized.
         ;; Most of the action in this event handler occurs here while we are waiting
         ;; for an UP event to occur. As long as we don't get it, keep erasing the
         ;; old zoom box and drawing a new one.

         ;; Safety measure (appears necessary if users are fast):
         IF NOT Ptr_Valid(info.rubberbox) THEN BEGIN
            Widget_Control, event.top, Set_UValue=info
            RETURN
         ENDIF

         ;; Get the window size
         IF NOT Obj_Valid(info.object) THEN $
            Message, 'Who killed the object ?'
         info.object->GetProperty, XSize=xsize, ysize=ysize

         ;; Erase the old zoom box, restore window content and draw new rectangle
         xs = 0 > event.x < (xsize-1)
         ys = 0 > event.y < (ysize-1)
         DrawRubberbox, info.rubberbox, xs, ys

         ;; Update status bar text
         ;; ### PRELIMINARY!! Show device coordinate values
         info.object->SetStatusText, '   X:'+String(xs,format='(i5)')  $
            +'  Y:'+String(ys,format='(i5)')
      END

      ELSE : BEGIN
         print,'Unknown event occured : '+thisevent+', IDL Version = '
         help,!Version,/Stru
         Widget_Control, event.top, Set_UValue=info

         RETURN
      END

   ENDCASE


   ;; Put the info structure back into its storage location.
   Widget_Control, event.top, Set_UValue=info

END




; -----------------------------------------------------------------------------
; MGS_Contour_Cleanup:
; Close the contour drawing window and destroy the object. This
; procedure is called from XManager if the window is closed or
; destroyed.

PRO MGS_Contour_Cleanup, tlb

   Widget_Control, tlb, Get_UValue=info, /No_Copy
   IF N_Elements(info) NE 0 THEN BEGIN
      Obj_Destroy, info.object
   ENDIF

END


; =============================================================================
; Object methods: Widget handling
; =============================================================================

; -----------------------------------------------------------------------------
; SaveAs:  (Event handler or callable)
; This method calls David Fanning's TVREAD program to capture the
; screen content of the drawing window and save it into a graphics
; file. The file type can be specified either as string argument or
; via keywords. The event argument is not used.

PRO MGS_Contour::SaveAs, filetype, event,   $
               BMP=bmp, PNG=png, GIF=gif, TIFF=tiff, JPEG=jpeg, PICT=pict, $
               filename=filename, nodialog=nodialog


   FORWARD_FUNCTION TVREAD

   ;; Error Handler
   Catch, theError
   IF self.debug THEN Catch, /Cancel
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error saving image to file!'
      RETURN
   ENDIF

   ;; Get window ID of draw widget
   WSet, self->GetWID()

   ;; Set filename
   basename = 'contour'
   nodialog = 0
   IF N_Elements(filename) GT 0 THEN BEGIN
      basename = filename
      nodialog = 1
   ENDIF 

   ;; Determine file type
   IF N_Elements(filetype) GT 0 THEN BEGIN
      CASE StrUpCase(filetype) OF
         'BMP' : bmp=1
         'GIF' : gif=1
         'PICT': pict=1
         'PNG' : png=1
         'JPEG': jpeg=1
         'TIFF': tiff=1
         ELSE: BEGIN
            self->ErrorMessage, 'Unknown file type!'
            RETURN
         END
      ENDCASE
   ENDIF

   ;; Get image and save it to file
   image = TVREAD(filename=basename,  $
                  BMP=Keyword_Set(bmp),  $
                  GIF=Keyword_Set(gif),  $
                  PICT=Keyword_Set(pict),  $
                  PNG=Keyword_Set(png),  $
                  JPEG=Keyword_Set(jpeg),  $
                  TIFF=Keyword_Set(tiff), $
                  nodialog=nodialog )


END


; -----------------------------------------------------------------------------
; SelectStyle:  (Event handler or callable)
; This method chooses a plot style for the contour plot, makes the
; appropriate settings in the object values and calls the Show method
; to redisplay the plot.
;    The style argument is a string with one of the values 'shaded',
; 'colorlines', 'bwlines', 'boxes'
; The event argument is not used.

PRO MGS_Contour::SelectStyle, theStyle, event

   ;; Error Handler
   Catch, theError
   IF self.debug THEN Catch, /Cancel
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error choosing a plot style!'
      RETURN
   ENDIF

   ;; Return if no style is selected
   IF N_Elements(theStyle) EQ 0 THEN RETURN

   ;; Do style settings
   CASE StrLowCase(theStyle) OF
      'shaded' : BEGIN
         self.lines = 0
         self.monochrome = 0
         self.boxes = 0
         self.image = 0
      END
      'colorlines' : BEGIN
         self.lines = 1
         self.monochrome = 0
         self.boxes = 0
         self.image = 0
      END
      'bwlines': BEGIN
         self.lines = 1
         self.monochrome = 1
         self.boxes = 0
         self.image = 0
      END
      'boxes' : BEGIN
         self.lines = 0
         self.monochrome = 0
         self.boxes = 1
         self.image = 0
      END
      'image' : BEGIN
         self.lines = 0
         self.monochrome = 0
         self.boxes = 0
         self.image = 1
      END
      ELSE: BEGIN
         self->ErrorMessage, 'Unknown style!'
         RETURN
      END
   ENDCASE

   ;; Redisplay plot with new settings
   self->Show

END


; -----------------------------------------------------------------------------
; ConfigurePS:  (Event handler or callable)
; This method calls David Fanning's postscript configuration tool
; including the font setup. Since the psconfig object stays "alive"
; afterwards, the settings will be remembered.

PRO MGS_Contour::ConfigurePS, event, _Extra=extra, NoFontInfo=NoFontInfo, $
               Cancel=cancel

   ;; Error Handler
   Catch, theError
   IF self.debug THEN Catch, /Cancel
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error configuring postscript device!'
      RETURN
   ENDIF

   ;; Determine group leader
   group_leader = self.tlb

   ;; Display font setup?
   fontinfo = ( Keyword_Set(NoFontInfo) EQ 0 )

   IF Obj_Valid(self.psconfig_obj) AND self.no_dialog EQ 0 THEN BEGIN
      ;; Set psconfig properties
      IF N_Elements(extra) GT 0 THEN $
         self.psconfig_obj->SetProperty, _Extra=extra

      ;; Call GUI of postscript configuration tool
      self.psconfig_obj->GUI, Group_Leader=group_leader, FontInfo=fontinfo, $
         Cancel=cancel
   ENDIF

END


; -----------------------------------------------------------------------------
; PrintPS:  (Event handler or callable)
; This method calls David Fanning's postscript configuration tool,
; opens a postscript file with the desired properties and reproduces
; the current graphic. Extra keywords are passed to the postscript
; configuration tool if the routine is called directly.
;   The fsc_psconfig object is permanently "alive" and thus remembers
; its settings between all calls.

PRO MGS_Contour::PrintPS, event, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF self.debug THEN Catch, /Cancel
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error creating postscript output!'
      RETURN
   ENDIF

   IF Obj_Valid(self.psconfig_obj) AND self.no_dialog EQ 0 THEN BEGIN
      ;; Set psconfig properties
      self->ConfigurePS, _Extra=extra, /NoFontInfo, Cancel=cancel
      IF Cancel THEN RETURN

      ;; Determine the keywords that should be passed to the device
      keywords = self.psconfig_obj->GetKeywords(FontType=fonttype)
   ENDIF ELSE BEGIN
      FontType = !P.Font
      keywords = { color:1, bits_per_pixel:8 }
   ENDELSE

   IF self.debug THEN BEGIN
      print,'Keywords to postscript device : '
      help,keywords, /Stru
   ENDIF
   ;; Set device to postscript and reproduce current graphics
   olddev = !D.Name
   oldp = !P
   Set_Plot, 'PS'
   Device, _Extra=keywords
   !P.Font = fonttype

   self->Show

   Set_Plot, olddev
   !P = oldp

END


; -----------------------------------------------------------------------------
; ConfigureMap:  (Event handler or callable)
; This method calls David Fanning's postscript configuration tool
; including the font setup. Since the psconfig object stays "alive"
; afterwards, the settings will be remembered.

PRO MGS_Contour::ConfigureMap, event, Cancel=cancel, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF self.debug THEN Catch, /Cancel
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error configuring postscript device!'
      RETURN
   ENDIF

   ;; Determine group leader
   group_leader = self.tlb

   IF Obj_Valid(self.mapsetup_obj) AND self.no_dialog EQ 0 THEN BEGIN
      ;; Set psconfig properties
      IF N_Elements(extra) GT 0 THEN $
         self.mapsetup_obj->SetProperty, _Extra=extra

      ;; Call GUI of map setup tool
      self.mapsetup_obj->GUI, Group_Leader=group_leader
   ENDIF

   ;; Retrieve keywords from object and store locally
   ;; **** NOTE: in the new MGS_BaseGUI concept, all properties
   ;; pertaining to the map setup shall be kept within the map setup
   ;; object and will be retrieved from there when needed. Hence, all
   ;; such properties as continentcolor, continentthick, etc. will be
   ;; removed from the mgs_contour object!!!

END


; -----------------------------------------------------------------------------
; Exit:  (Event handler or callable)
; This method destroys the object which will automatically delete all
; associated graphics windows.

PRO MGS_Contour::Exit, event, _EXTRA=e

   Obj_Destroy, self

END

; -----------------------------------------------------------------------------
; CreateWindow: (private)
; This method creates the drawing widget with its base and (later) the
; function buttons.

PRO MGS_Contour::CreateWindow

   ;; Don't do anything if current device is postscript
   IF !D.Name EQ 'PS' THEN RETURN

   ;; Create a top-level base for this program. No resizing of this base.
   self.tlb = Widget_Base(TLB_Frame_Attr=1, MBar=menuID)

   ;; Create the menu

   ;; 1. File menu
   self.MenuFileID = Widget_Button(menuID, Value='File')

   saveID = Widget_Button(self.MenuFileID,   $
                          Value='Save As ...',   $
                          /Menu)
   self.MenuFileSaveAsID = saveID

   dummy = Widget_Button(saveID, Value='Postscript File', $
                         Event_Pro='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'PrintPS' } )
   dummy = Widget_Button(saveID, Value='BMP File',  $
                         Event_Pro='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SaveAs', Argument:'BMP'} )
   dummy = Widget_Button(saveID, Value='GIF File',  $
                         Event_Pro='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SaveAs', Argument:'GIF'} )
   dummy = Widget_Button(saveID, Value='PICT File', $
                         Event_Pro='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SaveAs', Argument:'PICT'} )
   dummy = Widget_Button(saveID, Value='PNG File',  $
                         Event_Pro='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SaveAs', Argument:'PNG'} )
   dummy = Widget_Button(saveID, Value='JPEG File', $
                         Event_Pro='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SaveAs', Argument:'JPEG'} )
   dummy = Widget_Button(saveID, Value='TIFF File', $
                         Event_Pro='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SaveAs', Argument:'TIFF'} )

   self.MenuExitID = Widget_Button(self.MenuFileID,   $
                         Event_PRO='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'Exit' }, $
                         Value='Exit')

   ;; 2. Style menu
   MenuStyleID = Widget_Button(menuID, Value='Style')

   dummy = Widget_Button(MenuStyleID,   $
                         Event_PRO='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SelectStyle', Argument:'shaded' }, $
                         Value='shaded')

   dummy = Widget_Button(MenuStyleID,   $
                         Event_PRO='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SelectStyle', Argument:'colorlines' }, $
                         Value='colored lines')

   dummy = Widget_Button(MenuStyleID,   $
                         Event_PRO='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SelectStyle', Argument:'bwlines' }, $
                         Value='b/w lines')

   dummy = Widget_Button(MenuStyleID,   $
                         Event_PRO='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SelectStyle', Argument:'boxes' }, $
                         Value='boxes')

   dummy = Widget_Button(MenuStyleID,   $
                         Event_PRO='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'SelectStyle', Argument:'image' }, $
                         Value='image style')

   ;; 3. Options menu
   self.MenuOptionsID = Widget_Button(menuID, Value='Options')

   self.MenuPSConfigID = Widget_Button(self.MenuOptionsID,   $
                         Event_PRO='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'ConfigurePS' }, $
                         Value='Configure Postscript')

   self.MenuMapConfigID = Widget_Button(self.MenuOptionsID,   $
                         Event_PRO='MGS_Contour_EventHandler', $
                         UValue={ Object:self, Method:'ConfigureMap' }, $
                         Value='Configure Map Settings')

   ;; Create the draw widget
   drawbase = Widget_Base(self.tlb, Map=1)
   self.drawID = Widget_Draw(drawbase, XSize=self.wxsize, YSize=self.wysize, $
                             Retain=2, Button_Events=1, $
                             Event_Pro='MGS_CONTOUR_BUTTONEVENT')
   ;; Note: if the contour plot shall be made with object graphics,
   ;; add the following keywords:
   ;; Graphics_Level=2, Renderer=1, Retain=0, Expose_Events=1  (David Fanning)
   ;; (according to David, software rendering must be used because
   ;; hardware rendering is extremely slow)


   ;; Realize the widget
   Widget_Control, self.tlb, /Realize

   ;; Get the window ID
   ;; For object graphics, this will return the window object reference
   Widget_Control, self.drawID, Get_Value=theWindow
   self.wid = theWindow
   WSet, self.wid

   ;; Create the status bar
   self.StatusBaseID = Widget_Base(self.tlb, Row=1, Frame=2)
   self.StatusTextID = Widget_Label(self.StatusBaseID, /Align_Left, $
                                    Value=string('',format='(A40)'))

   ;; Set the title of the main window.
   Widget_Control, self.tlb, TLB_Set_Title='MGS_Contour'

   ;; Hide status bar if showstatus=0
   IF self.showstatus EQ 0 THEN $
      Widget_Control, self.StatusBaseID, Map=0

   ;; Store object reference in UValue of top level base
   Widget_Control, self.tlb,   $
      Set_UValue={ object:self, rubberbox:Ptr_New() }

   ;; Register program and set up event loop
   XManager, 'MGS_Contour', self.tlb, Cleanup='MGS_Contour_Cleanup', $
      /No_Block
;;      Group_Leader=group_leader, /No_Block

END


; -----------------------------------------------------------------------------
; GetWID:
; This method returns the window ID of the object's draw widget

FUNCTION MGS_Contour::GetWID

   RETURN, self.wid

END


; =============================================================================
; Object methods: General object functionality
; =============================================================================

; -----------------------------------------------------------------------------
; GetColorValue:
;    Set a drawing color: if given by name, use David Fanning's
; GetColor function to get a device independent color
; representation. If the color argument is numeric, the user must
; know the correct value (either a color table index or a long
; word for a decomposed color).
;    Specify index to load the color to a specific place in the color
; table (default is to use the top value minus one). The default
; keyword specifies a default color NAME to be used if the color
; argument is undefined (or not present).

FUNCTION MGS_Contour::GetColorValue, color, index=index, default=default

   result = !P.Color

   ;; Set default color table index
   IF N_Elements(index) EQ 0 THEN index = !D.Table_Size-1

   IF N_Elements(color) GT 0 THEN BEGIN
      argtype = Size(color, /TName)
      IF argtype EQ 'STRING' THEN BEGIN
         result = GetColor(color[0], index)
      ENDIF ELSE BEGIN
         result = color[0]
      ENDELSE
   ENDIF ELSE BEGIN
      IF N_Elements(default) GT 0 THEN $
         result = GetColor(default, index)
   ENDELSE

   RETURN, result
END


; -----------------------------------------------------------------------------
; GetNearestData:
; This method returns the value of the data on the display that is
; closest to the given x and y coordinate
; ### Maybe this method should be extended to use the full data range ?

FUNCTION MGS_Contour::GetNearestData, x, y

   wx = ( Where(*self.X GE x, cnt) )[0]
   IF cnt EQ 0 THEN wx = N_Elements(*self.X)-1
   wy = ( Where(*self.Y GE y, cnt) )[0]
   IF cnt EQ 0 THEN wy = N_Elements(*self.Y)-1

   RETURN, (*self.data)[wx,wy]

END


; -----------------------------------------------------------------------------
; IsGlobalMap:
; This function returns 1 if the current plot is a map and spans the
; globe. Otherwise 0 is returned.

FUNCTION MGS_Contour::IsGlobalMap

   ;; No map, return 0
   IF self.ismap EQ 0 THEN RETURN, 0

   ;; Limit reset to zero?
   IF Abs(Max(self.limit)) LT 1.E-3 THEN RETURN, 1

   ;; Global range in limit?
   RETURN, (self.limit[3]-self.limit[1]) GE 358. $
      OR (self.limit[2]-self.limit[0]) GE 177.5

END


; -----------------------------------------------------------------------------
; SetStatusText:
; This procedure changes the text in the status bar

PRO MGS_Contour::SetStatusText, newtext

   IF self.StatusTextID GT 0 THEN $
      Widget_Control, self.StatusTextID, Set_Value=newtext

END


; -----------------------------------------------------------------------------
; DrawColorBoxes:    (private)
;   This method produces filled polygons of the data. Coordinates are
; linearily interpolated to their edges.

PRO MGS_Contour::DrawColorBoxes, levels=levels, c_colors=c_colors, $
               min_valid=min_valid
;; (unused arguments) , data, X, Y, levels=levels, c_colors=c_colors, Min_Value=min_value

   ;; QUICK HACK: copy relevant information
   data = *self.data
   X = *self.X
   Y = *self.Y


   ;; Determine X and Y edges
   nx = N_Elements(X)
   ;; ### FIX
   IF X[nx-1] LT X[nx-2] THEN X[nx-1] = X[nx-1]+360.
   dx0 = X[1]-X[0]
   dx1 = X[nx-1]-X[nx-2]
   xedges = ( 0.5 * ( X + Shift(X,1) ) )[1:*]
   xedges = [  xedges[0]-dx0, xedges, xedges[nx-2]+dx1 ]

   ny = N_Elements(Y)
   dy0 = Y[1]-Y[0]
   dy1 = Y[ny-1]-Y[ny-2]
   yedges = ( 0.5 * ( Y + Shift(Y,1) ) )[1:*]
   yedges = [  yedges[0]-dy0, yedges, yedges[ny-2]+dy1 ]

   ;; Limit edges to valid lat coordinates if data is on map
   IF self.ismap THEN BEGIN
      yedges = ( yedges > (-90.) ) < 90.
   ENDIF

;; print,'xedges=',xedges
;; print,'yedges=',yedges

   maxcolorindex = N_Elements(c_colors)-1

   ;; Draw a frame around boxes only if less than 30 boxes in x and y
   drawframe = (nx LT 30) AND (ny LT 30)

   ;; Loop through all X's and Y's and draw filled polygons
   FOR i=0L,nx-1 DO BEGIN
      FOR j=0L,ny-1 DO BEGIN
         ;; Rectangle coordinates
         px = [ xedges[i], xedges[i+1], xedges[i+1], xedges[i], xedges[i] ]
         py = [ yedges[j], yedges[j], yedges[j+1], yedges[j+1], yedges[j] ]
         ;; Determine color value for this "pixel"
         cind = ( Where(levels GE data[i,j]) )[0] < maxcolorindex
         IF cind LT 0 THEN cind = maxcolorindex
         Polyfill, px, py, color=c_colors[cind]
         IF drawframe THEN PlotS, px, py, color=self.color
      ENDFOR
   ENDFOR

END

; -----------------------------------------------------------------------------
; DrawImage:    (private)
;   This method produces an image with colors corresponding to the
; data values.

PRO MGS_Contour::DrawImage, Min_Value=min_value

   data = *self.data

   ;; Find current plot position (after map set)
   position = [ !X.Window[0], !Y.Window[0], !X.Window[1], !Y.Window[1] ]

   ;; Transform data to byte array and interpolate (NOTE: Don't do
   ;; this for postscript output !!!)
   datadim = Size(data, /Dimensions)
   img = BytScl(Congrid(data, datadim[0]*10, datadim[1]*10, /Interp), $
                min=min_value, max=max(data), top=220) + 5

help,img

   img = img[*,0:350]
   tvimage, img, position=position, /Overplot


END

; -----------------------------------------------------------------------------
; DrawContourPlot:   (private)
;   This method produces a contour plot with the object settings. It
; is called from Show.

PRO MGS_Contour::DrawContourPlot, levels=levels, c_colors=c_colors, $
               filled=filled, min_valid=min_valid


   ;; Draw actual contours or boxes
   Contour, *self.data, *self.X, *self.Y, levels=levels, $
      fill=filled*(1-self.ismap),cell_fill=filled*self.ismap,  $
      c_colors=c_colors, Min_Value=min_valid, /Overplot
   
END


; -----------------------------------------------------------------------------
; SetupViewport:  (private)
;   This method calls the map setup or sets up the plot axes.

PRO MGS_Contour::SetupViewport, info=info

   ;; Set up map if requested or coordinate system otherwise
   ;; ### How do I set the background color for MAP_SET ???
; print,'### CURRENT LIMIT = ',self.limit
   IF self.ismap THEN BEGIN
      IF MAX(ABS(self.limit)) GT 1.E-3 THEN BEGIN
         map_set, 0, (self.limit[1]+self.limit[3])/2., limit=self.limit, $
            color=self.color    ; , background=backcolor
      ENDIF ELSE BEGIN
         MAP_SET, color=self.color ; , background=backcolor
      ENDELSE
   ENDIF ELSE BEGIN
      xmin = Min(*self.X, max=xmax)
      ymin = Min(*self.Y, max=ymax)
      Plot, [xmin,xmax], [ymin,ymax], /NoData, xstyle=1, ystyle=1,  $
         color=self.color, background=backcolor, charsize=self.charsize
   ENDELSE

END


; -----------------------------------------------------------------------------
; SetupColors:  (private)
;   This method restores the appropriate color table for the plot to
;   be displayed. Table=0 is the default table for contour plots,
;   table=1 is for image display.

PRO MGS_Contour::SetupColors, ctable=ctable

   info = *self.ctables[ctable]
   tvlct, info.r, info.g, info.b

END


; -----------------------------------------------------------------------------
; SetupContourParams:  (private)
;   This method determines the color table indices to be used, the
;   contour levels and the minimum valid data value.
; NOTE: min_valid may be undefined

PRO MGS_Contour::SetupContourParams, c_colors=c_colors, levels=levels, $
               filled=filled, min_valid=min_valid


   filled = 1-Keyword_Set(self.lines)

   IF Arg_Present(c_colors) THEN BEGIN
      IF Ptr_Valid(self.c_colors) THEN BEGIN
         c_colors = *self.c_colors
         ;; If numbers of levels and colors do not agree, interpolate
         ;; colors
         ;; #### (to be done)
      ENDIF ELSE BEGIN
         c_colors = indgen(N_Elements(*self.levels))+self.bottom
      ENDELSE
   ENDIF 

print,'$$SetupContourParams::C_COLORS=',c_colors   
   ;; ... or do we want monochrome contours? (only if not filled)
   IF filled EQ 0 AND self.monochrome THEN $
      c_colors = self.color
   

   levels = *self.levels
   ;; Add one extra level to achieve color filling below first
   ;; specified level. For consistency, do the same when drawing
   ;; contour lines.
   ;; ### Decrement value is random. Make sure it works for log
   ;; levels!
   ;; Need to draw contours in non-decomposed mode
   levels = [ -9.99E30, levels ]
   IF self.log THEN BEGIN
      levels[0] = 1.E-30
      min_valid = 1.E-30
   ENDIF


END


; -----------------------------------------------------------------------------
; AddOverlays:  (private)
;   This method adds continent outlines or filled continents if
;   desired and a colorbar.

PRO MGS_Contour::AddOverlays, info=info

   ;; Overlay continents for map plots
   IF self.ismap THEN BEGIN
      hires = (self.limit[2]-self.limit[0] LT 30.) $
         OR (self.limit[3]-self.limit[1] LT 30.)
      map_continents, color=self.continentcolor, $
         fill=self.continentfill, $
         thick=self.continentthick     ; , hires=hires
   ENDIF ELSE BEGIN
   ;; Redraw coordinate sytem for non-map plots
      Plot, [info.xmin,info.xmax], [info.ymin,info.ymax], /NoData, $
         xstyle=1, ystyle=1,  $
         color=self.color, charsize=self.charsize, /NoErase
   ENDELSE

END


; -----------------------------------------------------------------------------
; Annotate:  (private)
;   This method adds plot titles etc.

PRO MGS_Contour::Annotate

   ;; For demonstration purposes, we just add the variable name as title
   self.Variable->GetProperty, name=name, long_name=long_name

   thename = long_name
   IF thename EQ '' THEN thename = name

   XYOuts, 0.5, 0.95, thename, /Norm, color=self.color, $
      align=0.5, charsize=2.

END


; -----------------------------------------------------------------------------
; Show:
; This method displays the contour plot in a drawing widget

PRO MGS_Contour::Show

; print,'###show: ok=',self.ok
   ;; Test internal error condition. If unknown, call PrepareData
   ;; method
   IF self.ok LT 0 THEN BEGIN
      self->PrepareData
   ENDIF
   ;; Test again
   IF self.ok LE 0 THEN RETURN

   ;; Create draw widget or replace previous plot
   IF self.tlb LE 0 THEN self->CreateWindow

   ;; Device dependent settings
   isPrinter = (!D.Name EQ 'PS' OR !D.Name EQ 'PRINTER')

   ;; Make draw widget the active window (unless device is postscript)
   IF NOT isPrinter THEN WSet, self.wid

   ;; If we are producing a postscript graphic, force background to
   ;; white
   backcolor = self.backcolor
   IF !D.Name EQ 'PS' THEN backcolor = GetColor("white", 0)

   ;; Save axes settings
   oldX = !X
   oldY = !Y

   ;; Change axes settings for this plot/map_set
   ;; Increase thickness for printer output
   !X.Charsize=self.charsize
   !Y.Charsize=self.charsize
   !X.thick = self.axisthick + isPrinter*2.5
   !Y.thick = self.axisthick + isPrinter*2.5

   ;; Produce the plot
   self->SetupViewport, info=info
   IF NOT isPrinter THEN BEGIN
      Device, Get_Decomposed=theDecomposedState
      Device, Decomposed=0
   ENDIF
   self->SetupColors, ctable=self.image
   
   IF self.boxes THEN BEGIN
      self->SetupContourParams, c_colors=c_colors, levels=levels, min_valid=min_valid
      self->DrawColorBoxes, c_colors=c_colors, levels=levels, min_valid=min_valid
   ENDIF ELSE IF self.image THEN BEGIN
      self->SetupContourParams, min_valid=min_valid
      self->DrawImage, Min_Value=min_valid
   ENDIF ELSE BEGIN
      self->SetupContourParams, c_colors=c_colors, levels=levels, $
         filled=filled, min_valid=min_valid
      self->DrawContourPlot, c_colors=c_colors, levels=levels, $
         filled=filled, min_valid=min_valid
   ENDELSE

   self->AddOverlays, info=info
   self->Annotate
   IF NOT isPrinter THEN BEGIN
      Device, Decomposed=theDecomposedState
   ENDIF

   ;; Restore only charsize. Need to have Window tag for zooming !!
   !X.charsize = oldX.charsize
   !Y.charsize = oldY.charsize
   !X.thick = oldX.thick
   !Y.thick = oldY.thick
END


; -----------------------------------------------------------------------------
; FindLevels:
; This method determines the optimum contour levels for linear or
; logarithmic contouring

PRO MGS_Contour::FindLevels

   ;; Is number of levels fixed?
   IF self.nlevels GT 0 THEN BEGIN
      IF self.log THEN BEGIN
         tmpmin = ALOG10(self.minvalue > 1.0E-31)
         tmpmax = ALOG10(self.maxvalue > 1.0E-31)
         levels = lindgen(self.nlevels)  $
            * (tmpmax-tmpmin)/float(self.nlevels) + tmpmin
         levels = 10.^(levels < 30.)
      ENDIF ELSE BEGIN
         levels = lindgen(self.nlevels+1)  $
            * (self.maxvalue-self.minvalue)/float(self.nlevels) $
            + self.minvalue
      ENDELSE
   ENDIF ELSE BEGIN
      ;; Use optimum number of levels
      IF self.log THEN BEGIN
         levels = loglevels([self.minvalue, self.maxvalue], /Fine)
      ENDIF ELSE BEGIN
         levels = lindgen(16)*(self.maxvalue-self.minvalue)/15.+self.minvalue
      ENDELSE
   ENDELSE

print,'NLEVELS=',self.nlevels,'   log=',self.log
print,'Contour levels = ',levels
;;   IF Abs(Min(levels)-Max(levels)) LT 1.E-6 THEN $
;;      levels = findgen(15)

   IF Ptr_Valid(self.levels) THEN Ptr_Free, self.levels
   self.levels = Ptr_New(levels)


END


; -----------------------------------------------------------------------------
; PrepareData:
; This method analyses the LIMIT property and extracts the relevant
; portion of the variable data and the coordinates. Whenever the
; variable or the LIMIT is changed, this method must be called.

PRO MGS_Contour::PrepareData

   Catch, theError
   IF self.debug THEN Catch, /Cancel
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error preparing data for contour plot!'
      RETURN
   ENDIF

   ;; Error Checking
   IF NOT Obj_Valid(self.variable) THEN BEGIN
      self->ErrorMessage, 'Variable object is not valid!'
      RETURN
   ENDIF

   ;; Set error flag
   self.ok = 0

   ;; Extract the data array from the variable
   data = self.variable->GetData()

   ;; Get the dimension variables
   ;; ### Limitation: variable must be 2D
   self.variable->GetDimVar, 1, dim1
   self.variable->GetDimVar, 2, dim2

   ;; Check validity of dimension variables
   IF NOT Obj_Valid(dim1) THEN BEGIN
      self->ErrorMessage, '1. Dimension in variable object is not valid!'
      RETURN
   ENDIF
   IF NOT Obj_Valid(dim2) THEN BEGIN
      self->ErrorMessage, '2. Dimension in variable object is not valid!'
      RETURN
   ENDIF

   ;; Get dimension values (X and Y)
   X = dim1->GetData()
   Y = dim2->GetData()

   ;; Make sure dimensions are consistent
   dims = Size(data, /Dimensions)
   test1 = N_Elements(X)
   test2 = N_Elements(Y)
   IF (dims[0] NE test1) OR (dims[1] NE test2) THEN BEGIN
      self->ErrorMessage, 'Data and coordinate dimensions mismatch!'
      print, ' Data : ',dims, ' Coordinates : ',test1,test2
      RETURN
   ENDIF

   ;; Create working copy of limit for map display
   limit = self.limit

   ;; Map specialty: If no limit is given, it is assumed to be
   ;; -180,180 in longitude. Convert X values accordingly.
   ;; X coordinate also needs to be converted in some other
   ;; cases. ####
   ;; Dummy sort index
   sx = lindgen(N_Elements(X))
   IF self.ismap THEN BEGIN
      ;; Default global projection: depends on centerlon value.
      ;; If centerlon = 0, then use -180..180
      IF Abs(Max(limit)) LT 1.E-3 THEN $
         limit = [ -90., self.centerlon-180., 90., self.centerlon+180. ]

      ;; Make sure limit is map conform
      IF limit[1] GT limit[3] THEN limit[3] = limit[3]+360.
      limit[0] = (limit[0] > (-90.)) < limit[2]
      limit[2] = (limit[2] < 90.) > limit[0]
      limit[1] = (limit[1] > (-180.)) < limit[3]
      limit[3] = (limit[3] < 360.) > limit[1]
      ;; Need to store new limit for map_set
      self.limit = limit

      ;; Make X values consistent with map limit
      wlow = Where(X LT limit[1], cnt)
      IF cnt GT 0 THEN X[wlow] = X[wlow] + 360.
      whi = Where(X GT limit[3], cnt)
      IF cnt GT 0 THEN X[whi] = X[whi] - 360.

      ;; Make sure Y values do not reach 90 degrees
      wlow = Where(Y LE -90., cnt)
      IF cnt GT 0 THEN Y[wlow] = -89.9
      whi = Where(Y GE 90., cnt)
      IF cnt GT 0 THEN Y[whi] = 89.9

      ;; Sort X but keep sort index for use with data
      sx = Sort(X)
      X = X[sx]
   ENDIF

   ;; Test if LIMIT needs to be applied
   x0 = limit[1]
   x1 = limit[3]
   y0 = limit[0]
   y1 = limit[2]

   IF x0 LT x1 THEN BEGIN
      wx = Where( X GE x0 AND X LE x1, cnt )
      IF cnt GT 0 THEN BEGIN
         ;; extent wx by 1 element on each side if possible
         nx = N_Elements(wx)
         first = wx[0]-1
         IF first GE 0 THEN wx = [ first, wx ]
         ;; Last index is always N_Elements(X)-1
         ;; If field is global, add first element at end
         IF X[N_Elements(X)-1]-X[0] GT 350. THEN wx = [ wx, 0 ]
         ;; Cut out selected region
         X = X[wx]
         data = data[sx[wx], *]
      ENDIF ELSE BEGIN
         self->ErrorMessage, 'No data in given LIMIT range (X)!'
         RETURN
      ENDELSE
   ENDIF

   IF y0 LT y1 THEN BEGIN
      wy = Where( Y GE y0 AND Y LE y1, cnt )
      IF cnt GT 0 THEN BEGIN
         ;; extent wy by 1 element on each side if possible
         first = wy[0]-1
         IF first GE 0 THEN wy = [ first, wy ]
         ny = N_Elements(wy)
         last = wy[ny-1]+1
         IF last LT N_Elements(Y) THEN wy = [ wy, last ]
         ;; Cut out selected region
         Y = Y[wy]
         data = data[*, wy]
      ENDIF ELSE BEGIN
         self->ErrorMessage, 'No data in given LIMIT range (Y)!'
         RETURN
      ENDELSE
   ENDIF

   ;; Store finalized data in object
   Ptr_Free, self.data
   self.data = Ptr_New(data)
   Ptr_Free, self.X
   self.X = Ptr_New(X)
   Ptr_Free, self.Y
   self.Y = Ptr_New(Y)

   ;; Set status to ok
   self.ok = 1
END

; -----------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user.

PRO MGS_Contour::GetProperty, $
   Levels=levels,             $ ; The contour levels
   C_Colors=c_colors,         $ ; The contour colors
   XSize=xsize,               $ ; The window size of the drawing widget
   YSize=ysize,               $ ; dto.
   Centerlon=centerlon,       $ ; The central longitude for global map
   Limit=limit,               $ ; The range limit of X and Y
   Rubberboxcolor=rubberboxcolor, $ ; The color of the zoom rubberbox
   Report_Events=report_events,  $  ; Debugging aid
   CTable=ctable,             $ ; Current color table
   wid=wid,   $                 ; Window ID of drawing ID
   _Ref_Extra=extra             ; Inherited and future keywords
                                ;
                                ; Inherited keywords:
                                ; name      : The variable name
                                ; uvalue    : a user-defined value


   ;; Get properties from base object
   self->MGS_BaseObject::GetProperty, _Extra=extra

   Catch, theError
   IF self.debug THEN Catch, /Cancel
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving object properties!'
      RETURN
   ENDIF

   ;; Get scalar properties
   xsize = self.wxsize
   ysize = self.wysize

   centerlon = self.centerlon
   limit = self.limit

   ;; Get contour levels
   IF Arg_Present(levels) THEN BEGIN
      IF Ptr_Valid(self.levels) THEN $
         levels = *self.levels  $
      ELSE  $
         self->Undefine, levels
   ENDIF

   ;; Get contour colors
   IF Arg_Present(c_colors) THEN BEGIN
      IF Ptr_Valid(self.c_colors) THEN $
         levels = *self.c_colors  $
      ELSE  $
         self->Undefine, c_colors
   ENDIF

   rubberboxcolor = self.rubberboxcolor
   report_events = self.report_events

   wid = self.wid

   IF Arg_Present(ctable) THEN BEGIN
      IF self.image THEN ctable = *(self.ctables[1]) $
      ELSE ctable = *(self.ctables[0])
   ENDIF 

END


; -----------------------------------------------------------------------------
; SetProperty:
; This method sets specific object values. Derived objects may want to
; overwrite and extend this method to allow storing additional
; information.
; Notes:
; (1) Use negative nlevels to determine optimum number of contour
; levels automatically
; ### NEED TO IMPROVE automatic update of colortable when nlevels
; changes --- XColors should be able to return chosen colortable
; Currently, you need to call SetProperty twice: first any keywords
; which may change the number of color levels, then with the
; colortable keyword.


PRO MGS_Contour::SetProperty, $
   Levels=levels,             $    ; Contour levels
   NLevels=nlevels,           $    ; Number of contour levels for Findlevels
   MinValue=minvalue,         $    ; Minimum value to use in FindLevels
   MaxValue=maxvalue,         $    ; Maximum value to use in FindLevels
   Log=log,                   $    ; Use logarithmic contour intervals in
                                   ; Findlevels
   Color=color,               $    ; Main drawing color
   BackColor=backcolor,       $    ; Background color
   C_Colors=c_colors,         $    ; Contour colors
   Bottom=bottom,             $    ; Bottom most colortable value
   CTable=ctable,             $    ; The index of the colortable to be used
   Map=map,                   $    ; Change to map mode (1) or back (0) (### EXPERIMENTAL)
   Centerlon=centerlon,       $    ; Set center longitude for global map
   Limit=limit,               $    ; Restrict data range (set to 0 to show full range)
   Lines=lines,               $    ; Plot contour lines (1) or fill (0)
   Monochrome=monochrome,     $    ; Draw contours in foreground and background color
   Boxes=boxes,               $    ; Draw colored boxes
   Image_Style=image_style,   $    ; Show data as image
   Windowsize=windowsize,     $    ; Size of the widget draw window
   Charsize=charsize,         $    ; Change character size
   AxisThick=axisthick,       $    ; Line thickness of axes
   ContinentColor=continentcolor, $; Color for continents in map plots
   ContinentFill=continentfill,   $; Fill mode for continents
   ContinentThick=continentthick, $; Thickness of continent outlines
   Rubberboxcolor=rubberboxcolor, $; Change the color of the zoom rubberbox
   Debug=debug,               $    ; Turn debugging flag on/off
   Report_events=report_events, $  ; Toggle switch for event status report
   ShowStatus=showstatus,     $    ; Show (1) or hide (0) status bar
   _Ref_Extra=extra                ;
                                   ; Inherited keywords:
                                   ; name      : The variable name
                                   ; no_copy   : Don't keep local copy
                                   ;             of uvalue
                                   ; no_dialog : Don't display
                                   ;             interactive dialogs
                                   ; uvalue    : a user-defined value



   Catch, theError
   IF self.debug THEN Catch, /Cancel
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error setting object properties!'
      RETURN
   ENDIF

   ;; Set Properties of base object
   self->MGS_BaseObject::SetProperty, _Extra=extra

   ;; Set contour levels
   IF N_Elements(levels) GT 0 THEN BEGIN
      ;; Make sure levels are increasing
      theselevels = levels[sort(levels)]
      *self.levels = theselevels
   ENDIF

   ;; Define contour intervals logarithmically or linear
   IF N_Elements(log) GT 0 THEN BEGIN
      self.log = Keyword_Set(log)
   ENDIF

   ;; Define minimum or maximum value for contour levels
   IF N_Elements(minvalue) GT 0 THEN $
      thismin = minvalue[0]  $
   ELSE $
      thismin = self.minvalue

   IF N_Elements(maxvalue) GT 0 THEN $
      thismax = maxvalue[0]  $
   ELSE $
      thismax = self.maxvalue

   ;; Check consistency of min and maxvalue
   IF thismin GE thismax THEN BEGIN
      self->ErrorMessage, 'MinValue GE MaxValue! ('+String(thismin)+','  $
         +String(thismax)+')'
      RETURN
   ENDIF
   self.minvalue = thismin
   self.maxvalue = thismax

   ;; Define contour intervals logarithmically or linear
   IF N_Elements(nlevels) GT 0 THEN BEGIN
      self.nlevels = ( nlevels[0] > 2 ) < 58
   ENDIF

   ;; Main drawing color
   IF N_Elements(color) GT 0 THEN BEGIN
      self.color = self->GetColorValue(color)
   ENDIF
   IF N_Elements(backcolor) GT 0 THEN BEGIN
      self.backcolor = self->GetColorValue(backcolor, index=0)
   ENDIF

   ;; Contour colors: set -1 to delete
   IF N_Elements(c_colors) GT 0 THEN BEGIN
      IF Ptr_Valid(self.c_colors) Then Ptr_Free, self.c_colors
      IF c_colors[0] GE 0 THEN $
         self.c_colors = Ptr_New( (Fix(c_colors)>0) < (!D.Table_Size-1) )
   ENDIF

   ;; Bottom value of colortable
   IF N_Elements(bottom) GT 0 THEN BEGIN
      self.bottom = ( bottom[0] > 0 ) < (!D.Table_Size-1)
   ENDIF

   ;; Colortable
;; ### mgs NEED TO IMPROVE handling of user input: currently only
;; table 0 can be changed, and it is not possible to directly provide
;; r, g, and b values!!
   IF N_Elements(ctable) GT 0 THEN BEGIN
      ;; save old color table
      tvlct, oldr, oldg, oldb, /get
      ;; Determine current number of levels
      ncolorlevels = self.nlevels
      IF ncolorlevels LT 0 THEN ncolorlevels = N_Elements(*self.levels) > 3
      IF ctable LT 0 THEN BEGIN
         XColors, bottom=self.bottom, ncolors=ncolorlevels
         thetable = -1
      ENDIF ELSE BEGIN
         thetable = ctable < 40
         loadct, thetable, bottom=self.bottom, $
            ncolors=ncolorlevels, /Silent
      ENDELSE
      Ptr_Free, self.ctables[0]
      tvlct, r, g, b, /get
      self.ctables[0] = Ptr_New( { r:r, g:g, b:b, index:thetable } )
      ;; restore old color table
      tvlct, oldr, oldg, oldb
   ENDIF

   ;; Switch map mode on or off  (### ONLY EXPERIMENTAL !!)
   IF N_Elements(map) GT 0 THEN $
      self.ismap = Keyword_Set(map)

   ;; Set central longitude for global map
   IF N_Elements(centerlon) GT 0 THEN BEGIN
      self.centerlon = ( centerlon[0] > (-180.)) < 360.
      ;; Need to update data if global view is active
      IF (Abs(Max(self.limit)) LT 1.E-3) $
          OR (Abs(self.limit[3]-self.limit[1]-360.) LT 1.E-3) THEN BEGIN
         self.ok = -1
         self.limit = 0.
      ENDIF
   ENDIF

   ;; Limit data range
   IF N_Elements(limit) GT 0 THEN BEGIN
      IF N_Elements(limit) NE 4 AND N_Elements(limit) NE 1 THEN BEGIN
         self->ErrorMessage, 'Limit must be a 4-element vector!'
         RETURN
      ENDIF
      self.limit = limit

      ;; Indicate that data needs to be extracted again
      self.ok = -1
   ENDIF

   ;; Set contour mode to lines
   IF N_Elements(lines) GT 0 THEN $
      self.lines = Keyword_Set(lines)

   ;; Monochrome contours ?
   IF N_Elements(monochrome) GT 0 THEN $
      self.monochrome = Keyword_Set(monochrome)

   ;; Colored boxes instead of contours?
   IF N_Elements(boxes) GT 0 THEN BEGIN
      self.boxes = ( Keyword_Set(boxes) AND (Keyword_Set(lines) EQ 0) )
      IF self.boxes THEN BEGIN
         self.lines = 0
         self.image = 0
      ENDIF
   ENDIF

   ;; Image style instead of contours?
   IF N_Elements(image_style) GT 0 THEN BEGIN
      self.image = ( Keyword_Set(image_style) AND $
                     (Keyword_Set(boxes) EQ 0) AND (Keyword_Set(lines) EQ 0) )
      IF self.image THEN BEGIN
         self.lines = 0
         self.boxes = 0
      ENDIF
   ENDIF

   IF N_Elements(windowsize) GT 0 THEN BEGIN
      IF N_Elements(windowsize) NE 2 THEN BEGIN
         self->ErrorMessage, 'Windowsize keyword must have two elements!'
         RETURN
      ENDIF

      self.wxsize = long(windowsize[0]) > 0
      self.wysize = long(windowsize[1]) > 0
   ;; Destroy widgets and let widget be built again
      IF self.tlb GT 0 THEN $
         Widget_Control, self.tlb, /Destroy
   ENDIF

   ;; Character size for title and labels
   IF N_Elements(charsize) GT 0 THEN BEGIN
      self.charsize = ( charsize[0] > 0.3 ) < 10.
   ENDIF

   ;; Line thickness of axes
   IF N_Elements(axisthick) GT 0 THEN BEGIN
      self.axisthick = axisthick[0]
   ENDIF

   IF N_Elements(continentcolor) GT 0 THEN BEGIN
      self.continentcolor = self->GetColorValue(continentcolor, $
                                                index=!D.Table_Size-4)
   ENDIF

   IF N_Elements(continentfill) GT 0 THEN BEGIN
      self.continentfill = continentfill[0]
   ENDIF

   IF N_Elements(continentthick) GT 0 THEN BEGIN
      self.continentthick = continentthick[0]
   ENDIF

   ;; Change status bar
   IF N_Elements(showstatus) GT 0 THEN BEGIN
      self.showstatus = Keyword_Set(showstatus)
      IF self.StatusBaseID GT 0 THEN BEGIN
         Widget_Control, self.StatusBaseID, Map=self.ShowStatus
      ENDIF
   ENDIF

   ;; Character size for title and labels
   IF N_Elements(rubberboxcolor) GT 0 THEN BEGIN
      self.rubberboxcolor = self->GetColorValue( rubberboxcolor, index=!D.Table_Size-2 )
   ENDIF

   ;; Debugging tools
   IF N_Elements(debug) GT 0 THEN $
      self.debug = Keyword_Set(debug)
   IF N_Elements(report_events) GT 0 THEN $
      self.report_events = Keyword_Set(report_events)

   ;; (for pointer values)
   ;; IF N_Elements(gattr) GT 0 THEN BEGIN
   ;;   IF Ptr_Valid(self.gattr) THEN Ptr_Free, self.gattr
   ;;   self.gattr = Ptr_New(gattr)
   ;;   IF ChkStru(gattr, 'title') THEN self.title = gattr.title
   ;; ENDIF

END


; -----------------------------------------------------------------------------
; Cleanup:
; This method frees all data stored in the object.

PRO MGS_Contour::CLEANUP

   ;; Destroy any GUI
   ;; self->KillGUI

   ;; Delete variable object (??? maybe not the smartest idea
   ;; always???)
   IF Obj_Valid(self.variable) THEN BEGIN
      ;; ### Limitation to 2D variable !!
      self.variable->GetDimVar, 1, thisX
      Obj_Destroy, thisX
      self.variable->GetDimVar, 2, thisY
      Obj_Destroy, thisY
      Obj_Destroy, self.variable
   ENDIF

   ;; Destroy postscript configurator
   IF Obj_Valid(self.psconfig_obj) THEN Obj_Destroy, self.psconfig_obj
   IF Obj_Valid(self.mapsetup_obj) THEN Obj_Destroy, self.mapsetup_obj

   ;; Free all pointers
   Ptr_Free, self.data
   Ptr_Free, self.X
   Ptr_Free, self.Y
   Ptr_Free, self.levels
   Ptr_Free, self.c_colors
   Ptr_Free, self.ctables

   ;; Destroy widgets
   IF self.tlb GT 0 THEN $
      Widget_Control, self.tlb, /Destroy

   ;; Call parent's cleanup method
   self->MGS_BaseObject::Cleanup

END




; -----------------------------------------------------------------------------
; Init:
;   This method initializes the file object.

FUNCTION MGS_Contour::INIT, $
      thisVariable, $
      levels=levels,  $
      nlevels=nlevels,    $
      minvalue=minvalue,  $
      maxvalue=maxvalue,  $
      log=log,  $
      color=color,        $
      backcolor=backcolor, $
      c_colors=c_colors,  $
      bottom=bottom,      $
      ctable=ctable, $
      ismap=ismap,        $
      centerlon=centerlon, $
      limit=limit,        $
      lines=lines,        $
      monochrome=monochrome,  $
      boxes=boxes,   $
      image_style=image_style,   $
      windowsize=windowsize,  $
      charsize=charsize,   $
      axisthick=axisthick, $
      continentcolor=continentcolor,  $
      continentfill=continentfill, $
      continentthick=continentthick,  $
      rubberboxcolor=rubberboxcolor,  $
      debug=debug,         $
      report_events=report_events,    $
      _Ref_Extra=extra                 ; For inherited and future keywords
                                       ;
                                       ; Inherited keywords:
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value



   ;; Initialize parent object first
   IF not self->MGS_BaseObject::Init(_Extra=extra) THEN RETURN, 0

   Catch, theError
   IF Keyword_Set(Debug) THEN Catch,/Cancel
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error initializing object!'
      RETURN, 0
   ENDIF

   ;; Set internal error condition to unknown
   self.ok = -1

   ;; Check validity of variable
   IF N_Elements(thisVariable) EQ 0 THEN BEGIN
      self->ErrorMessage, 'Must supply variable object!'
      RETURN, 0
   ENDIF
   IF NOT Obj_Valid(thisVariable) THEN BEGIN
      self->ErrorMessage, 'Variable argument must be object reference!'
      RETURN, 0
   ENDIF
   thisVariable->GetProperty, ndims=ndims
   IF ndims NE 2 THEN BEGIN
      self->ErrorMessage, 'Variable object must be 2-dimensional!'
      RETURN, 0
   ENDIF

   ;; Make sure, variable dimvars are correct
   ;; ...  ###

   ;; Store variable in object
   self.variable = thisVariable

   ;; Store other properties
   IF N_Elements(levels) GT 0 THEN BEGIN
      self.levels = Ptr_New(levels)
   ENDIF ELSE BEGIN
      self.levels = Ptr_New()
   ENDELSE

   IF N_Elements(nlevels) GT 0 THEN BEGIN
      self.nlevels = nlevels[0]
   ENDIF ELSE BEGIN
      IF N_Elements(levels) GT 0 THEN $
         self.nlevels = N_Elements(levels)  $
      ELSE  $
         self.nlevels = -1
   ENDELSE


   ;; Get min and max of data or store minvalue and maxvalue
   ;; properties
   ;; #### Should already take LIMIT range into account !!
   tmpmin = self.variable->Min()
   tmpmax = self.variable->Max()
   IF N_Elements(minvalue) GT 0 THEN BEGIN
      self.minvalue = minvalue[0]
   ENDIF ELSE BEGIN
      self.minvalue = tmpmin
   ENDELSE
   IF N_Elements(maxvalue) GT 0 THEN BEGIN
      self.maxvalue = maxvalue[0]
   ENDIF ELSE BEGIN
      self.maxvalue = tmpmax
   ENDELSE


   ;; Store contour properties
   self.log = Keyword_Set(log)

   ;; Set main drawing color and background color device independently
   self.color = self->GetColorValue(color, default="black")
   self.backcolor = self->GetColorValue(backcolor, index=0, default="white")

   IF N_Elements(c_colors) GT 0 THEN BEGIN
      self.c_colors = Ptr_New(c_colors)
   ENDIF ELSE BEGIN
      self.c_colors = Ptr_New()
   ENDELSE

   IF N_Elements(bottom) GT 0 THEN BEGIN
      self.bottom = (bottom[0] > 0) < 255
   ENDIF ELSE BEGIN
      self.bottom = 2
   ENDELSE

   ;; Colortable
   ;; Store curren table
   tvlct, oldr, oldg, oldb, /get
;; ### mgs NEED TO IMPROVE handling of user input: currently only
;; table 0 can be changed, and it is not possible to directly provide
;; r, g, and b values!!
   IF N_Elements(ctable) EQ 0 THEN ctable = 27

   ;; Determine current number of levels
   ncolorlevels = self.nlevels
   IF ncolorlevels LT 0 THEN BEGIN
      IF Ptr_Valid(self.levels) THEN BEGIN
         ncolorlevels = N_Elements(*self.levels) > 3
      ENDIF ELSE BEGIN
         ncolorlevels = 16
      ENDELSE 
   ENDIF 
   IF ctable LT 0 THEN BEGIN
      XColors, bottom=self.bottom, ncolors=ncolorlevels
      thetable = -1
   ENDIF ELSE BEGIN
      thetable = ctable < 40
      loadct, thetable, bottom=self.bottom, $
         ncolors=ncolorlevels, /Silent
   ENDELSE
   tvlct, r, g, b, /get
   self.ctables[0] = Ptr_New( { r:r, g:g, b:b, index:thetable } )

   loadct, 1
   tvlct, r, g, b, /get
   r = reverse(r)
   g = reverse(g)
   b = reverse(b)
   self.ctables[1] = Ptr_New( { r:r, g:g, b:b, index:1 } )
   ;; restore old color table
   tvlct, oldr, oldg, oldb

   self.ismap = Keyword_Set(ismap)

   IF N_Elements(centerlon) GT 0 THEN BEGIN
      self.centerlon = ( centerlon[0] > (-180.)) < 360.
   ENDIF ELSE BEGIN
      self.centerlon = 0.
   ENDELSE

   IF N_Elements(limit) EQ 4 THEN BEGIN
      self.limit = limit
   ENDIF ELSE BEGIN
      self.limit = 0.
   ENDELSE

   self.lines = Keyword_Set(lines)
   self.monochrome = Keyword_Set(monochrome)
   self.boxes = Keyword_Set(boxes)
   self.image = Keyword_Set(image_style)
   ;; Consistency check
   IF self.lines THEN BEGIN
      self.boxes = 0
      self.image = 0
   ENDIF
   IF self.boxes THEN self.image = 0

   IF N_Elements(windowsize) GT 0 THEN BEGIN
      IF N_Elements(windowsize) NE 2 THEN BEGIN
         self->ErrorMessage, 'Windowsize keyword must have two elements!'
         RETURN, 0
      ENDIF

      self.wxsize = long(windowsize[0]) > 0
      self.wysize = long(windowsize[1]) > 0
   ;; Destroy widgets and let widget be built again
      IF self.tlb GT 0 THEN $
         Widget_Control, self.tlb, /Destroy
   ENDIF ELSE BEGIN
      Device, Get_Screen_Size=scrsize
      self.wxsize = ceil(scrsize[0]/2.)
      self.wysize = ceil(scrsize[1]/2.)
   ENDELSE


   IF N_Elements(charsize) GT 0 THEN BEGIN
      self.charsize = ( charsize > 0.3 ) < 10.
   ENDIF ELSE BEGIN
      self.charsize = 1.0
   ENDELSE

   IF N_Elements(axisthick) GT 0 THEN BEGIN
      self.axisthick = axisthick[0]
   ENDIF ELSE BEGIN
      self.axisthick = 1.5
   ENDELSE

   self.continentcolor = self->GetColorValue(continentcolor, $
                                             index=!D.Table_Size-4, $
                                             default="white")

   IF N_Elements(continentfill) GT 0 THEN BEGIN
      self.continentfill = continentfill[0]
   ENDIF ELSE BEGIN
      self.continentfill = 0
   ENDELSE

   IF N_Elements(continentthick) GT 0 THEN BEGIN
      self.continentthick = continentthick[0]
   ENDIF ELSE BEGIN
      self.continentthick = 1.5
   ENDELSE

   self.rubberboxcolor = self->GetColorValue(rubberboxcolor, $
                                             index=!D.Table_Size-2, $
                                             default="white")

   ;; Initialize postscript configuration tool
   self.psconfig_obj = Obj_New('fsc_psconfig', /European)

   ;; Initialise map setup tool
   self.mapsetup_obj = Obj_New('mgs_mapsetup')

   self.debug = Keyword_Set(debug)
   self.report_events = Keyword_Set(report_events)

   ;; Initialize widget properties
   ;; (but only if current device is not postscript)
   IF !D.Name NE 'PS' THEN BEGIN
      self.tlb = 0
      self.drawID = 0
      self.wid = -1
      Device, Get_Screen_Size=scrsize

      self.StatusBaseID = 0
      self.StatusTextID = 0
      self.ShowStatus = 0   ;; Default: Don't display status bar (wrong location) ###
   ENDIF


   ;; If levels are not given explicitely, find them
;; print,'#### min,max = ',self.minvalue,self.maxvalue
   IF NOT Ptr_Valid(self.levels) THEN self->FindLevels

   ;; If current device is postscript, set filename explicitely
   ;; Not sure why, but appears to be necessary!!!   #########
   IF !D.Name EQ 'PS' THEN BEGIN
      Device, filename='idl.ps'
   ENDIF

   ;; Display contour plot
   self->Show

   ;; Destroy object if current device is postscript
   IF !D.Name EQ 'PS' THEN BEGIN
      Device, /Close
      print, 'Postscript file produced.'
      self->Cleanup
      RETURN, 0
   ENDIF

   RETURN, 1
END



; -----------------------------------------------------------------------------
; MGS_Contour__Define:
; This is the object definition for a generic object. It
; inherits from MGS_BaseObject the abilities to set and query an
; object name and a uvalue. The base object also provides a general
; method for display of error messages which can be directed to a
; message dialog or to the log screen via the no_dialog flag.

PRO MGS_Contour__Define

   objectClass = { MGS_Contour, $ ; The object class
           variable:Obj_New(), $  ; The variable object with the complete data set
           data:Ptr_New(),     $  ; The data actually displayed
           X:Ptr_New(),        $  ; The X coordinates actually displayed
           Y:Ptr_New(),        $  ; The Y coordinates actually displayed
           levels:Ptr_New(),   $  ; The contour levels
           nlevels:0,          $  ; The number of contour levels
           minvalue:0.0D,      $  ; Minimum range
           maxvalue:0.0D,      $  ; Maximum range
           log:0,              $  ; Logarithmic contour intervals
           color:0L,           $  ; Main drawing color
           backcolor:0L,       $  ; Background color
           c_colors:Ptr_New(), $  ; Contour colors (index values)
           bottom:0,           $  ; Index of first color to use for contours
           ctables:PtrArr(2),  $  ; Colortables (0=contour plots, 1=image display)
           ismap:0,            $  ; Contour is a geographic map
           limit:[0.,0.,0.,0.],$  ; X and Y boundaries for plotting or map
           centerlon:0.,       $  ; Central longitude for global(!) map
           lines:0,            $  ; Draw contour as lines
           monochrome:0,       $  ; Draw contours as monochrome lines
           boxes:0,            $  ; Draw colored boxes instead of contours
           image:0,            $  ; Draw image instead of contours
           charsize:0.0,       $  ; Character size for title and labels
           axisthick:0,        $  ; Line thickness for axes
           continentcolor:0L,  $  ; Color for continents
           continentfill:0L,   $  ; Fill type for continents
           continentthick:0.,  $  ; Line thickness for continent outlines
           ;; internal error condition
           ok:0,               $
           ;; widget identifiers
           tlb:0L,             $  ; The top level base
           MenuFileID:0L,      $  ; The file menu (button)
           MenuFilePrintID:0L, $  ; The File-Print button
           MenuFileSaveAsID:0L,$  ; The File-SaveAs button
           MenuExitID:0L,      $  ; The File-Exit button
           MenuOptionsID:0L,   $  ; The Options menu
           MenuPSConfigID:0L,  $  ; The 'configure postscript' button
           MenuMapConfigID:0L,  $  ; The 'configure map settings' button
           drawID:0L,          $  ; The draw widget
           wid:0,              $  ; The window ID for the drawing widget
           wxsize:0L,          $  ; The widget size (width)
           wysize:0L,          $  ; (height)
           StatusBaseID:0L,    $  ; The ID of the status bar base widget
           StatusTextID:0L,    $  ; The ID of the status text widget
           ShowStatus:0,       $  ; Display (or hide) status bar
           ;; other stuff
           rubberboxcolor:0L,  $  ; Color value of the zoom rubberbox (default !P.color)
           psconfig_obj:Obj_New(),  $ ; The postscript configuration tool
           mapsetup_obj:Obj_New(),  $ ; The map configuration tool
           ;; debugging aids
           report_events:0L,   $  ; Report all events in detail for debugging
           ;;; heritage
           INHERITS MGS_BaseObject  $ ; provides basic general properties
     }

END
