;+
; NAME:
;   MGS_Field
;
; PURPOSE:
;
;   This is a widget object that defines a simple input field which
;   can accept a variety of input types, together with a label. It is
;   a complete rewrite of David Fanning's FSC_Field program and builds
;   on the MGS_BaseGUI class hierarchy.
;      The MGS_Field widget can be configured to accept only numeric
;   data, only positive values, or only a certain number of
;   decimals. You can also specifiy a character mask to allow only
;   specific characters to be entered or even a regular expression
;   (the usefulness of which is not entirely clear to me yet ;-). 
;      As the original FSC_Field program, this widget corrects certain
;   display deficiencies of IDL's library routine CW_Field.
;      This widget can be used as a blocking or non-blocking
;   stand-alone widget or as a compound widget (depending on the call
;   to the GUI method; see MGS_BaseGUI), and it inherits from
;   MGS_BaseGUI the ability to reset itself if you define the 'Reset'
;   button. See MGS_InputMask for an example dialog using several
;   MGS_Field objects as compound widgets. 
;
; AUTHOR:
;
;   Dr. Martin Schultz
;   Max-Planck-Institut fuer Meteorologie
;   Bundesstr. 55, D-20146 Hamburg
;   email: martin.schultz@dkrz.de
;
; CATEGORY:
;   Widget objects
;
; TYPICAL CALLING SEQUENCE:
;   thefield = Obj_New('MGS_Field', 3.14159, min_valid=0., $
;                      max_valid=10.)
;   thefield->GUI, /block
;   print, 'The new value is ',thefield->GetValue()
;   Obj_Destroy, thefield
;
; PARAMETERS:
;   None.
;
; KEYWORDS:
;   See the respective method headers. See Init for the keywords to be
;   used with the Obj_New() function.
;
; NOTES:
;   (1) Set the debug level to 3 if you want detailed output regarding
;   event handling or field validation (obj->setproperty,debug=3).
;   (2) This program served as a testbed to try out various concepts
;   for MGS_BaseGUI. In learning how to handle numeric or text input,
;   I created the concepts of updatemode, the reset functionality, and
;   several other ideas that are now available in the BaseGUI object.
;   (3) It is important to distinguish between the "object value" and
;   the "widget value" in this program. The "object value" (value and
;   oldvalue fields in BaseGUI) always contains a validated
;   information of the data type that you requested, whereas the
;   "widget value" (theText field in this object) reflects the
;   keystrokes that you type. If the updatemode is 2 ("immediate"),
;   the two values should be consistent at all times, otherwise there
;   may be a difference which will be resolved if you leave the text
;   widget focus (updatemode=1, "auto") or if you call the
;   UpdateObject method (updatemode=0, "never"). 
;
; ACKNOWLEDGEMENTS:
;   I owe a great deal of this program to David Fanning who invented
;   the right style of handling widget events in IDL and who
;   introduced me to IDL objects.
;
; MODIFICATION HISTORY:
;   mgs, 04 Apr 2001: Version 1.0 released
;       based on David Fanning's FSC_Field program as of 19 Jan 2001
;   mgs, 19 Apr 2001: - improved validation (now allows single '-' and
;                       treats it as '0')
;   mgs, 25 Apr 2001: - major adaptations in concert with new BaseGUI
;                       object 
;   mgs, 29 May 2001: - several bug fixes for treatment of NaN's
;-
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################
;


; -----------------------------------------------------------------------------
; MoveTab:
;   This method advances the input focus to the next editable field if
; the TAB key is pressed.

PRO MGS_Field::MoveTab
   IF  NOT Widget_Info(self.tabnext, /Valid_ID) THEN RETURN
   Widget_Control, self.tabnext, /Input_Focus
   Widget_Control, self.tabnext, Get_Value=theText
   theText = theText[0]
   Widget_Control, self.tabnext, Set_Text_Select=[0,StrLen(theText)]
END 


; -----------------------------------------------------------------------------
; SetTabNext:
;   This method stores the widget ID of the "next" input field

PRO MGS_Field::SetTabNext, nextID
   self.tabnext = nextID
END


; -----------------------------------------------------------------------------
; SetFocus:
;   This method sets the input focus to the object's text widget

PRO MGS_Field::SetFocus

   IF NOT Widget_Info(self.textID, /Valid_ID) THEN RETURN

   Widget_Control, self.textID, /Input_Focus

END


; -----------------------------------------------------------------------------
; GetTextID:
;   This method returns the ID of the text widget of the compound widget.

FUNCTION MGS_Field::GetTextID
   RETURN, self.textID
END


; -----------------------------------------------------------------------------
; Resize:
;    This method resizes the widget by making the text widget fit the new size.

PRO MGS_Field::Resize, newsize
   l = Widget_Info(self.labelID, /Geometry)
   Widget_Control, self.textID, Scr_XSize = (newsize - l.scr_xsize)

   ;; Set the text widget sensitivitiy.
   Widget_Control, self.textID, Sensitive=1-Keyword_Set(self.nonsensitive)
END


; -----------------------------------------------------------------------------
; Geometry:
;    This method returns the geometry of the compound widget.

FUNCTION MGS_Field::Geometry
   RETURN, Widget_Info(self.tlb,/Geometry)
END


; -----------------------------------------------------------------------------
; UpdateObject:  (private)
;    This method retrieves the current value of the text widget (a
; string), converts it to the object value's datatype and stores it in
; the object's value field.

PRO MGS_Field::UpdateObject

   IF NOT Widget_Info(self.tlb, /Valid_ID) THEN RETURN

   ;; Get the current contents of text widget. Validate it.
   Widget_Control, self.textID, Get_Value=newText
   newText = newText[0]

   ;; Convert to requested data type and validate
   newvalue = self->ConvertStringToValue(newtext, result=result)

   ;; Store new value in object's value field
   IF StrUpCase(result) EQ 'OK' THEN self->SetValue, newvalue

END


; -----------------------------------------------------------------------------
; GetState:
;   This method returns the actual value and datatype of the input field.

;FUNCTION MGS_Field::GetState

;   RETURN, { value:self->GetValue(), datatype:self.datatype }

;END


; -----------------------------------------------------------------------------
; SetValue:
;    This method stores a new value in the object. We need to
; overwrite the BaseGUI SetValue method because we must also set the
; string property theText to the new value.

PRO MGS_Field::SetValue, value, ok=ok

   self->MGS_BaseGUI::SetValue, value, ok=ok

   IF ok THEN BEGIN
      IF self.datatype EQ 'STRING' THEN self.theText = String(*self.value) $ 
         ELSE self.theText = StrTrim(*self.value,2)
      self->Show     ;; need to redraw the text widget again
   ENDIF 

END


; -----------------------------------------------------------------------------
; ConvertStringToValue:
;    This method converts a string argument (typically the current
; value of the text widget) to the object's datatype and validates this
; value.
;    The result keyword will contain either 'OK' or an error condition
; such as 'NO_VALID_DATA'.

FUNCTION MGS_Field::ConvertStringToValue, testvalue, result=result

   ;; fail safe default result
   result = 'NO_VALID_DATA'
   retval = *self.undefined

   ;; Error Handling
   ON_IOERROR, CatchIt

   IF N_Elements(testvalue) EQ 0 THEN BEGIN 
      self->ErrorMessage, 'testvalue must be defined'
   ENDIF ELSE BEGIN
      testarg = testvalue
   ENDELSE 

   ;; Strings can be returned as input unless we need to apply a
   ;; regular expression filter
   IF self.datatype EQ 'STRING' THEN BEGIN
      IF self.regex EQ '' THEN BEGIN
         result = 'OK'
         retval = testarg
      ENDIF ELSE BEGIN
         retval = self->ValidateRegex(testarg)
         IF retval EQ '' AND testarg NE '' THEN result = 'INVALID_STRING' $
            ELSE result = 'OK'
      ENDELSE 
      RETURN, retval
   ENDIF

   ;; For other data types, test for NULL string (this returns 0 in
   ;; IDL without causing an error, however, we don't think '' is a
   ;; valid number).

   IF testarg EQ '' THEN RETURN, retval

   ;; Attempt two conversions:
   ;; 1st convert to actual data type
   ;; 2nd convert to "largest in class" to allow for range checking
   ;; NOTES:
   ;; (1) A single '+' or '-' would cause an IO error (let's catch it)
   ;; (2) Unsigned long64 has no test for negative values
   
   IF testarg EQ '+' OR testarg EQ '-' THEN testarg = '0'

   CASE self.datatype OF
      'BYTE':    retval = Fix(testarg)
      'INT':     retval = Fix(testarg)
      'LONG':    retval = Long(testarg)
      'LONG64':  retval = Long64(testarg)
      'UINT':    retval = UInt(testarg)
      'ULONG':   retval = ULong(testarg)
      'ULONG64': retval = ULong64(testarg)
      'FLOAT' :  retval = Float(testarg)
      'DOUBLE':  retval = Double(testarg)
   ENDCASE
   CASE self.datatype OF
      'BYTE':    tstval = Long64(testarg)
      'INT':     tstval = Long64(testarg)
      'LONG':    tstval = Long64(testarg)
      'LONG64':  tstval = Long64(testarg)
      'UINT':    tstval = Long64(testarg)
      'ULONG':   tstval = Long64(testarg)
      'ULONG64': tstval = ULong64(testarg)
      'FLOAT' :  tstval = Double(testarg)
      'DOUBLE':  tstval = Double(testarg)
   ENDCASE

   ;; If we arrive here, the actual conversion went ok. Unless we
   ;; detect a range error now, this is the result status
   result = 'OK'

   retval = self->ValidateNumber(retval, testval, result=result)
   RETURN, retval

CatchIt:
   retval = *self.undefined
   result = 'NO_VALID_DATA'

   RETURN, retval

END


; -----------------------------------------------------------------------------
; SetDataType:
;    This method determines the datatype and gen(eral) type of an
; argument and stores that information in the respective object
; property fields. The result is a boolean value indicating the
; success of this operation.

FUNCTION MGS_Field::SetDataType, arg

   result = 0

   dataType = Size(arg, /TNAME)
   CASE dataType OF
      'BYTE'   : BEGIN
         genType = 'UNSIGNED'
         dataType = 'INT'
         value = Fix(value)
         self->ErrorMessage, 'BYTE data not supported. Value will be converted to INT.', $
            /Warning
      END
      'INT'    : genType = 'INTEGER'
      'LONG'   : genType = 'INTEGER'
      'LONG64' : genType = 'INTEGER'
      'UINT'   : genType = 'UNSIGNED'
      'ULONG'  : genType = 'UNSIGNED'
      'ULONG64': genType = 'UNSIGNED'
      'FLOAT'  : genType = 'FLOAT'
      'DOUBLE' : genType = 'FLOAT'
      'STRING' : genType = 'STRING'
      ELSE     : BEGIN
         Message, 'Data type ' + dataType + ' is not supported. Returning.'
         RETURN, result
      ENDELSE 
   ENDCASE

   ;; If positive flag is set and data is of gentype integer, pretend
   ;; it is unsigned
   IF self.positive AND genType EQ 'INTEGER' THEN genType = 'UNSIGNED'

   self.dataType = dataType
   self.gentype = genType

   RETURN, 1

END


; -----------------------------------------------------------------------------
; TestRegex:
;   This method tests the validity of a regular expression by trying
; it out.

FUNCTION MGS_Field::TestRegEx, expression

   ;; Error Handler (this is the clue)
   CATCH, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Regular expression not valid'
      RETURN, 0
   ENDIF

   void = StRegex('dummy', expression)

   RETURN, 1
END


; -----------------------------------------------------------------------------
; ValidateString:  (private)
;   This function returns a string that matches the regular expression
; defined in the regex field or it extracts those characters defined
; in the character_mask field. Regex takes precedence over
; character_mask. If neither regex nor character_mask are defined, the
; input string is returned unchanged.

FUNCTION MGS_Field::ValidateString, value, result=result

   ;; Error Handling.
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      Catch, /Cancel
;      self->ErrorMessage, 'Error validating input field value'
;      RETURN, ''
;   ENDIF

   result = 'OK'

   ;; Return if string is empty
   IF value EQ '' THEN RETURN, ''

   ;; Test for regular expression first
   IF self.regex NE '' THEN BEGIN

;;      result = StRegex(value, self.regex, /Extract, Fold_Case=self.fold_case)

   ENDIF ELSE IF self.character_mask NE '' THEN BEGIN
      ;; Extract valid characters
      asbyte = Byte(value)
      len = N_Elements(asbyte)
      bytemask = Byte(self.character_mask)
      IF self.fold_case THEN $
         bytemask = Byte(StrUpCase(self.character_mask)+StrLowCase(self.character_mask))
      FOR i=0L, len-1 DO BEGIN
         wv = Where(bytemask EQ asbyte[i], cnt)
         IF cnt GT 0 THEN BEGIN
            IF N_Elements(retval) EQ 0 THEN retval = asbyte[i] $
               ELSE retval = [ retval, asbyte[i] ]
         ENDIF
      ENDFOR
      IF N_Elements(retval) GT 0 THEN retval = String(retval) ELSE retval = ''

   ENDIF ELSE BEGIN
      ;; Return input string unaltered
      retval = value
   ENDELSE

   RETURN, retval
END


; -----------------------------------------------------------------------------
; ValidateNumber:  (private)
;    This function checks if numeric input is consistent with the
; object settings (data type range, min_valid and max_valid
; properties). Value and tstval are principally identical, except that
; tstval is always of the largest possible datatype in the general
; type, so that it can be used for range checking.  

FUNCTION MGS_Field::ValidateNumber, value, tstval, result=result

   result = 'OK'
   retval = value
   IF N_Elements(tstval) EQ 0 THEN tstval=value

   ;; Out-of-datatype-range check: retval and tstval must be equal
   IF self.gentype EQ 'FLOAT' THEN BEGIN
      IF finite(tstval) THEN BEGIN
         max_allowed = (machar()).xmax
         IF tstval LT -max_allowed OR tstval GT max_allowed THEN BEGIN
            retval = *self.undefined
            result = 'VALUE_EXCEEDS_DATATYPE_LIMIT'
         ENDIF 
      ENDIF ELSE IF NOT self.allow_nan THEN BEGIN
         retval = *self.undefined
         result = 'VALUE_EXCEEDS_DATATYPE_LIMIT(NaN)'
      ENDIF 
   ENDIF ELSE BEGIN
      IF tstval NE value THEN BEGIN
         retval = *self.undefined
         result = 'VALUE_EXCEEDS_DATATYPE_LIMIT'
      ENDIF 
   ENDELSE

   ;; Test unsigned types for negative values
   IF self.gentype EQ 'UNSIGNED' THEN BEGIN
      IF tstval LT 0 THEN BEGIN
         retval = *self.undefined
         result = 'NEGATIVE_VALUE_FOR_UNSIGNED_TYPE'
      ENDIF 
   ENDIF

   ;; Test for range
   IF N_Elements(*self.min_valid) GT 0 THEN BEGIN
      IF tstval LT *self.min_valid THEN BEGIN
         retval = *self.undefined
         result = 'VALUE_OUT_OF_RANGE'
      ENDIF 
   ENDIF
   IF N_Elements(*self.max_valid) GT 0 THEN BEGIN
      IF tstval GT *self.max_valid THEN BEGIN
         retval = *self.undefined
         result = 'VALUE_OUT_OF_RANGE'
      ENDIF 
   ENDIF

   RETURN, retval

END


; -----------------------------------------------------------------------------
; TestNumericInput: (private)
;    This function eliminates illegal characters from a string that represents
; a number. The return value is a properly formatted string that can be turned into
; an INT, LONG, FLOAT, or DOUBLE value.
;    This method is used to react on text events, i.e. after the user
; types a character or portion of the string, or after she deletes something. 

FUNCTION MGS_Field::TestNumericInput, value

   ;; Error Handling.
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      Catch, /Cancel
;      self->ErrorMessage, 'Error validating input field value'
;      RETURN, ''
;   ENDIF

   ;; Return if string is empty
   IF value EQ '' THEN RETURN, ''

   ;; Special case: If allow_nan is true, check for it
   IF self.allow_nan THEN BEGIN
      IF StrLowCase(value) EQ 'nan' OR StrLowCase(value) EQ 'n' $
         OR StrLowCase(value) EQ 'na' THEN RETURN, 'NaN'
   ENDIF

   ;; Make working copy of input string as byte array, convert to lower case
   asbyte = Byte(StrLowCase(value))
   len = N_Elements(asbyte)

   ;; Set initial flag values
   havedigit = 0
   havedecimal = 0
   haveexponent = 0
   previous_was_exponent = 0
   nexpdigits = 0    ; number of digits in exponent (must not exceed 3)

   ;; Compose set of valid characters for first character in test string
   CASE self.gentype OF
      'INTEGER' :  allch = '0123456789+-'
      'UNSIGNED' : allch = '0123456789+'
      'FLOAT' :    allch = '0123456789+-.'
      'HEX' :      allch = '0123456789abcdef'    ;; *** HEX not supported yet in object!
      ELSE : BEGIN
         self->ErrorMessage, 'Cannot validate this type as a number ('+self.gentype+')'
      ENDELSE 
   ENDCASE
   testbytes = Byte(allch)
;; print,'### value, allch:',string(asbyte),',',allch
   ;; Loop over characters of input string and add valid characters to
   ;; retval
   FOR i=0L, len-1 DO BEGIN
      wv = Where(testbytes EQ asbyte[i], cnt)
      IF cnt GT 0 AND nexpdigits LE 3 THEN BEGIN
         IF N_Elements(retval) EQ 0 THEN retval = asbyte[i] $
         ELSE retval = [ retval, asbyte[i] ]
         ;; Update flags
         IF self.gentype EQ 'FLOAT' THEN BEGIN
            IF wv[0] LE 9 THEN BEGIN
               havedigit = 1    ; '0..9'
               IF haveexponent THEN nexpdigits = nexpdigits+1
            ENDIF 
            IF asbyte[i] EQ 46B THEN havedecimal = 1    ; '.'
            IF asbyte[i] EQ 100B OR asbyte[i] EQ 101B THEN BEGIN
               haveexponent = 1 ; 'd' or 'e'
               previous_was_exponent = 1
            ENDIF 
         ENDIF
      ENDIF

      ;; Change character mask for second character:
      ;; INTEGER and UNSIGNED may not contain '+' or '-' after first
      ;; character
      ;; For gentype FLOAT, things are a little more complicated ;-)
      IF i EQ 0 THEN BEGIN
         CASE self.gentype OF
            'INTEGER' :  allch = '0123456789'
            'UNSIGNED' : allch = '0123456789'
            'FLOAT' :    BEGIN
               allch = '0123456789'
               ;; Allow '.' if not already entered
               IF NOT havedecimal THEN allch = allch+'.'
               ;; Allow exponent if first character was a digit
               IF havedigit THEN allch = allch+'de'
            END 
         ENDCASE
         testbytes = Byte(allch)

      ENDIF ELSE BEGIN

         ;; Change character mask for all other characters:
         ;; Need only consider float values
         IF self.gentype EQ 'FLOAT' THEN BEGIN
            allch = '0123456789'
            ;; First choice: has exponent been entered?
            IF haveexponent THEN BEGIN
               ;; Allow '+' or '-' if exponent character was the
               ;; previous character
               IF previous_was_exponent THEN allch = allch+'+-'
               previous_was_exponent = 0
            ENDIF ELSE BEGIN
               ;; No exponent character so far, thus it is valid if a
               ;; digit has been entered
               IF havedigit THEN allch = allch+'de'
               ;; Allow '.' if not already entered
               IF NOT havedecimal THEN allch = allch+'.'
            ENDELSE
            testbytes = Byte(allch)
         ENDIF
      ENDELSE

   ENDFOR

   ;; Check if retval string is empty
   IF N_Elements(retval) GT 0 THEN retval = String(retval) ELSE retval = ''
   IF self.debug EQ 3 THEN print,'validatenumber ### value,retval = ',value,' ',retval
   ;; We now have a numeric string that is syntactically
   ;; correct. However, the number may still be out of range or
   ;; otherwise "corrupted". Therefore, test the conversion to the
   ;; object data type and check if the retval is valid
;;   test = self->ReturnValue(result)
   test = self->ConvertStringToValue(retval, result=errorstring)
   IF self.debug EQ 3 THEN $
      print,'validatenumber ### test,result,retval = ',test,' ',errorstring,' ',retval

   IF errorstring NE 'OK' THEN BEGIN
      self->ErrorMessage,'Invalid number: '+errorstring
      RETURN, ''
   ENDIF ELSE BEGIN
      RETURN, StrTrim(retval,2)
   ENDELSE 

END


; -----------------------------------------------------------------------------
; Validate:

FUNCTION MGS_Field::Validate, arg, result=result

   ;; Error Handling.
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      Catch, /Cancel
;      self->ErrorMessage, 'Error validating input field value'
;      RETURN, 'NO_VALID_DATA'
;   ENDIF

   result = 'INVALID_DATA'

   ok = self->SetDataType(arg)

   IF NOT ok THEN RETURN, *self.undefined

   ;; Validate the input value depending on whether it is a string or
   ;; a number
   IF self.gentype EQ 'STRING' THEN BEGIN
      retValue = self->ValidateString(arg, result=result)
   ENDIF ELSE BEGIN
      retValue = self->ValidateNumber(arg, result=result)
   ENDELSE 

   RETURN, retValue

END


; -----------------------------------------------------------------------------
; ValidateRegex:
;    This function filters string input with the regular
; expression. It is called from GetValue or from TextEvents if the
; input field looses its focus.

FUNCTION MGS_Field::ValidateRegex, value

   ;; Error Handling.
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error validating input field value'
      RETURN, 'NO_VALID_DATA'
   ENDIF

   ;; Validate the input value depending on whether it is a string or
   ;; a number
   IF self.gentype NE 'STRING' THEN RETURN, value

   result = StRegex(value, self.regex, /Extract, Fold_Case=self.fold_case)

   RETURN, result

END


; -----------------------------------------------------------------------------
; HandleFocusEvents:
;   This method reacts to WIDGET_KBRD_FOCUS events, i.e. the user
; clicks on a different widget or X window, or presses the TAB or
; ENTER key.

FUNCTION MGS_Field::HandleFocusEvents, event
 
   ;; Gaining focus: no action
;   IF event.enter EQ 1 THEN BEGIN
;      RETURN, 0
;   ENDIF 

   ;; Loosing focus: if the text field has a regular expression
   ;; assigned, make sure it is valid  
   IF event.enter EQ 0 THEN BEGIN
      ;; Make sure object value reflects widget state
      IF self.updatemode GT 0 THEN self->UpdateObject

      IF self.datatype EQ 'STRING' AND self.regex NE '' THEN BEGIN
         IF self.debug GT 1 THEN print,'**** loosing focus: ...validate ...', $
            self.name,event.id,event.handler

         ;; Get current widget contents and make sure it is valid
         Widget_Control, self.textID, Get_Value=newText
         validtext = self->ValidateRegex(newText[0])
         IF validtext NE newText[0] THEN BEGIN
            self->ErrorMessage, ['String does not match regular expression!', $
                                 'expression='+self.regex]
            ;; Set focus back to text field to force correction
            Widget_Control, event.id, /Input_Focus
         ENDIF 
      ENDIF
 
      ;; We might want to pass information about this event to other
      ;; objects 
      theMessage = { object:self, eventtype:'LOOSING_FOCUS' }
      RETURN, theMessage

   ENDIF

   RETURN, 0

END


; -----------------------------------------------------------------------------
; HandleInsertEvents:
;   This method reacts to text insertion events (WIDGET_TEXT_CH and
; WIDGET_TEXT_STR).

FUNCTION MGS_Field::HandleInsertEvents, event
 
   ;; Get the previous and current contents of text widget. Validate it.
   previousText = self.theText
   Widget_Control, self.textID, Get_Value=newText
   textLocation = Widget_Info(event.id, /Text_Select)
   newText = newText[0]
; print,'### textlocation = ',textLocation
   IF self.datatype NE 'STRING' THEN BEGIN
      validText = self->TestNumericInput(newText)
   ENDIF ELSE BEGIN
      validText = self->ValidateString(newText)
   ENDELSE 
   IF self.debug EQ 3 THEN $
      print,'insert ### newtext,validtext = ',newtext,' ',validtext

   ;; If it is valid, leave it alone. If not, go back to previous text.

   IF validText NE newText AND validText NE 'NaN' THEN BEGIN
      Widget_Control, self.textID, Set_Value=previousText, $
         Set_Text_Select=[textLocation[0]-1,0]
   ENDIF ELSE BEGIN
      self.theText = validText
      self->Show, location=[textLocation[0],0]
   ENDELSE

   ;; We don't need to pass this event back to anyone
   RETURN, 0

END


; -----------------------------------------------------------------------------
; HandleDeleteEvents:
;   This method reacts to character deletions in the text field
; (WIDGET_TEXT_DEL) 

FUNCTION MGS_Field::HandleDeleteEvents, event

   ;; Get the current contents of text widget. Validate it.

   Widget_Control, self.textID, Get_Value=newText
   textLocation = Widget_Info(event.id, /Text_Select)
   newText = newText[0]
;; print,'### NEWTEXT, location=',newtext,textlocation
   IF self.datatype NE 'STRING' THEN BEGIN
      validText = self->TestNumericInput(newText)
   ENDIF ELSE BEGIN
      validText = self->ValidateString(newText)
   ENDELSE 

   IF self.debug EQ 3 THEN $
      print,'delete_text ### newtext,validtext = ',newtext,' ',validtext
   ;; Load the valid text.
   Widget_Control, self.textID, Set_Value=validText
;;      Set_Text_Select=[textLocation[0],0]
   self.theText = validText
   self->Show, location=[textLocation[0],0]

   ;; We don't need to pass this event back to anyone
   RETURN, 0

END


; -----------------------------------------------------------------------------
; TextEvents:
;    The event handler method for the text widget of the compound widget.

FUNCTION MGS_Field::TextEvents, event

   ;; Error Handling.
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      Catch, /Cancel
;      self->ErrorMessage, 'Error dealing with text events'
;      RETURN, 0
;   ENDIF


   ;; What kind of event is this?

   eventCategory = Tag_Names(event, /Structure_Name)

;; print,'###TEXTEVENT: ',eventcategory
;; print,'###value, datatype,thetext:',*self.value,self.datatype,self.thetext
   CASE StrUpCase(eventCategory) OF
      ;; Gaining or loosing focus
      'WIDGET_KBRD_FOCUS' : BEGIN
         RETURN, self->HandleFocusEvents(event)
      END

      ;; Change the cursor position or select part of the string
      'WIDGET_TEXT_SEL' : BEGIN
      END 

      ;; Insertion of a single character
      'WIDGET_TEXT_CH' : BEGIN
         ;; TAB or RETURN key
         IF event.ch EQ 9B OR event.ch EQ 10B THEN BEGIN
            self->MoveTab
            RETURN, 0   ;; event
         ENDIF
         RETURN, self->HandleInsertEvents(event)
      END 

      ;; Insertion of a string
      'WIDGET_TEXT_STR' : BEGIN
         RETURN, self->HandleInsertEvents(event)
      END

      ;; Delete a character or string
      'WIDGET_TEXT_DEL' : BEGIN
         RETURN, self->HandleDeleteEvents(event)
      END

      ELSE: BEGIN
         self->ErrorMessage, 'Unknown event catagory : '+ $
            eventCategory
      ENDELSE 
   ENDCASE 

   RETURN, 0
END


; -----------------------------------------------------------------------------
; SetEdit:
;   Make text field editable

PRO MGS_Field::SetEdit, editvalue

   ;; Error Handling.

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage
      RETURN
   ENDIF

   IF N_ELements(editvalue) EQ 0 THEN editvalue = 1
   Widget_Control, self.textID, Editable=editvalue
END


; -----------------------------------------------------------------------------
; SetSensitive:
;    Make text widget sensitive to input

PRO MGS_Field::SetSensitive, value

   ;; Error Handling.

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage
      RETURN
   ENDIF

   IF N_Elements(value) EQ 0 THEN value = 1
   Widget_Control, self.textID, Sensitive=value
END 


; -----------------------------------------------------------------------------
; Show:
;   This method updates the widget contents (the text field) if the
; GUI is currently on display. Use the location keyword to set the
; cursor position (see Widget_Control for detailed info).

PRO MGS_Field::Show, location=location

;; print,'###',routine_name(/caller)
   IF NOT Widget_Info(self.tlb, /Valid_ID) THEN RETURN

;   theText = String(*self.value)
;   IF self.datatype NE 'STRING' THEN theText = StrTrim(theText,2)
   theText = self.theText
   IF N_Elements(location) EQ 0 THEN location = [StrLen(theText),0]
;; print,'###SHOW: thetext, location = ',thetext,' ',location
   Widget_Control, self.textID, $
      Set_Value=theText, $
      Set_Text_Select=location

;   self.theText = theText
   ;; No compound widgets, so no need to call inherited Show method

END


; -----------------------------------------------------------------------------
; BuildGUI:  (private)
; This method builds the graphical user interface. Here, just a simple
; compound widget for text input is added to the empty frame provided
; by the generic BaseGUI Object.
; NOTE: 
; This method also duplicates the theText field so that it can later
; be restored if the dialog is cancelled

PRO MGS_Field::BuildGUI

   ;; Create the widgets.
   ;; The label
   self.labelID = Widget_Label( self.layoutID, $
                                Value=self.labeltext, $
                                Font=self.defaultfont, $
                                Scr_XSize=self.labelsize, $
                                UValue=self)

   ;; The text widget
   ;; Note: the text field is left blank here and will be updated by
   ;; the call to the Show method from BaseGUI->GUI
   scr_xsize = 0
   scr_ysize = 0
   self.textID = Widget_Text( self.layoutID, $ 
                              Value='', $
                              XSize=self.xsize, $
                              YSize=1, $
                              Scr_XSize=scr_xsize, $
                              Scr_YSize=scr_ysize, $
                              Font=self.fieldfont, $
                              All_Events=1, $
                              KBRD_Focus_Events=1, $  
                              Event_Pro='MGS_BaseGUI_Widget_Events', $
                              UValue={Method:"TextEvents", Object:self}, $
                              Editable=1-self.noedit )

   IF Ptr_Valid(self.value) THEN BEGIN
      theText = String(*self.value)
   ENDIF ELSE BEGIN
      theText = ''
   ENDELSE 
   IF self.datatype NE 'STRING' THEN theText = StrTrim(theText,2)
   self.theText = theText

END


; -----------------------------------------------------------------------------
; SetWidgetFonts:  (private)
; This method sets the widget font names. We need to overwrite the
; geneirc method because we added an extra font qualifier for the text
; widget.

PRO MGS_Field::SetWidgetFonts, $
     reset=reset,            $  ; Set this keyword to use default values if
                                ; the respective ...font keyword is not given
     widget_fieldfont=widget_fieldfont,    $  ; Font for the text widget
     _Extra=extra               ; This takes care of inherited keywords:
                                ;    widget_defaultfont
                                ;    widget_labelfont


   ;; Call inherited method
     self->MGS_BaseGUI::SetWidgetFonts, $
        reset=reset, $
        _Extra=extra

   IF N_Elements(widget_fieldfont) EQ 0 THEN BEGIN
      IF Keyword_Set(reset) THEN self.fieldfont = self.defaultfont
   ENDIF ELSE IF widget_fieldfont EQ '' THEN BEGIN
      self.fieldfont = self.defaultfont
   ENDIF ELSE BEGIN
      self.fieldfont = widget_fieldfont
   ENDELSE

END


; -----------------------------------------------------------------------------
; GetProperty:
;    This method allows you to obtain various properties of the
; compound widget via output keywords.
; NOTE:
; (1) Use the GetValue method to determine the current value of the field

PRO MGS_Field::GetProperty, $
   CR_Only=cr_only, $        ; Set this keyword if you only want Carriage Return events.
   DataType=datatype, $      ; The datatype of the compound widget.
   Decimal=decimal, $        ; The number of digits to the right of the decimal point in FLOAT numbers.
   Digits=digits, $          ; The number of digits permitted in INTERGERVALUE and LONGVALUE numbers.
   Allow_NaN=allow_nan, $    ; text widget shows 'NaN' if number is not finite
   Regex=regex,  $                  ; A regular expression that the input string must fulfill
   Character_Mask=character_mask, $ ; A list of valid characters
   Fold_case=fold_case, $           ; Ignore case for string validation
   Fieldfont=fieldfont, $
   Labeltext=labeltext, $
   Labelsize=labelsize, $
   NoEdit=noedit, $          ; Setting this keywords makes the text widget non-editable.
   NonSensitive=nonsensitive, $   ; Setting this keywords makes the text widget non-sensitive.
   Undefined=undefined, $    ; The "value" of any undefined value.
   _Ref_Extra=extra


   ;; Get properties from base object
   self->MGS_BaseGUI::GetProperty, _Extra=extra

   ;; Error Handling
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error retrieving object properties'
      RETURN
   ENDIF

   ;; Get the properties.
   cr_only = self.cr_only
   datatype = self.datatype
   decimal = self.decimal
   digits = self.digits
   allow_nan = self.allow_nan
   regex = self.regex
   character_mask = self.character_mask
   fold_case = self.fold_case
   fieldfont=self.fieldfont
   labeltext = self.labeltext
   labelsize = self.labelsize
   noedit = self.noedit
   nonsensitive = self.nonsensitive
   undefined = *self.undefined

END


; -----------------------------------------------------------------------------
; SetProperty:
;    This method allows you to set various properties of the compound
; widget.
; NOTES:
; (1) The font keywords take only effect if widget is rebuilt (another
;     call to GUI)
; (2) Set min_valid or max_valid to '' or if you want to remove either
; (3) You can change the value only with the SetValue method

PRO MGS_Field::SetProperty, $
   CR_Only=cr_only, $         ; Set this keyword if you only want Carriage Return events.
   Decimal=decimal, $         ; Number of digits to the right of the decimal point in FLOAT values..
   Digits=digits, $           ; Number of digits permitted in INTEGER values.
   Regex=regex,  $                  ; A regular expression that the input string must fulfill
   Character_Mask=character_mask, $ ; A list of valid characters
   Fold_case=fold_case, $           ; Ignore case for string validation
   Allow_NaN=allow_nan, $     ; Text widget shows 'NaN' if number is not finite
   Min_Valid=min_valid, $     ; minimum valid number
   Max_Valid=max_valid, $     ; maximum valid number
   Fieldfont=fieldfont, $     ; The font name for the text field
   Labeltext=labeltext, $     ; The text to go on the Label Widget.
   LabelSize=labelsize, $     ; The X screen size of the Label Widget.
   NoEdit=noedit, $           ; Setting this keywords makes the text widget non-editable.
   NonSensitive=nonsensitive, $  ; Setting this keywords makes the text widget non-sensitive.
   Scr_XSize=scr_xsize, $     ; The X screen size of the text widget.
   Scr_YSize=scr_ysize, $     ; The Y screen size of the text widget.
   Undefined=undefinded, $    ; Set to "value" of undefined value.
   XSize=xsize, $             ; The X size of the Text Widget.
   _Extra=extra


   ;; Set Properties of base object
   self->MGS_BaseGUI::SetProperty, _Extra=extra

   ;; Error Handling
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error setting object properties'
      RETURN
   ENDIF

   ;; Is widget active and needs updating?
   isactive = Widget_Info(self.tlb, /Valid_ID)

   ;; Set the properties, if needed.

   IF N_Elements(cr_only) NE 0 THEN self.cr_only = cr_only
   IF Keyword_Set(decimal)THEN self.decimal = decimal
   IF Keyword_Set(digits)THEN self.digits = digits
   IF N_Elements(allow_nan) THEN BEGIN
      self.allow_nan = Keyword_Set(allow_nan)
   ENDIF
   IF N_Elements(regex) THEN BEGIN
      ;; Test regular expression if provided
      IF regex NE '' THEN BEGIN
         IF self->TestRegEx(regex) THEN self.regex = String(regex)
      ENDIF ELSE self.regex = regex
   ENDIF
   IF N_Elements(character_mask) THEN BEGIN
      self.character_mask = String(character_mask)
   ENDIF
   IF N_Elements(fold_case) THEN BEGIN
      self.fold_case = Keyword_Set(fold_case)
   ENDIF
   IF N_Elements(labelsize) NE 0 THEN BEGIN
      self.labelsize = labelsize
      IF isactive THEN Widget_Control, self.labelID, XSize=labelsize
   ENDIF
   IF N_Elements(scr_xsize) NE 0 THEN BEGIN
      self.scr_xsize = scr_xsize
      IF isactive THEN Widget_Control, self.textID, Scr_XSize=scr_xsize
   ENDIF
   IF N_Elements(scr_ysize) NE 0 THEN BEGIN
      self.scr_ysize = scr_ysize
      IF isactive THEN Widget_Control, self.textID, Scr_YSize=scr_ysize
   ENDIF
   IF N_Elements(labeltext) NE 0 THEN BEGIN
      self.labeltext = labeltext
      IF isactive THEN Widget_Control, self.labelID, Set_Value=self.labeltext
   ENDIF
   IF N_Elements(xsize) NE 0 THEN BEGIN
      self.xsize = xsize
      IF isactive THEN Widget_Control, self.textID, XSize=xsize
   ENDIF
   IF N_Elements(noedit) THEN BEGIN
      self.noedit = Keyword_Set(noedit)
      IF isactive THEN Widget_Control, self.textID, Editable=1-self.noedit
   ENDIF
   IF N_Elements(nonsensitive) THEN BEGIN
      self.nonsensitive = Keyword_Set(nonsensitive)
      IF isactive THEN Widget_Control, self.textID, Sensitive=1-self.nonsensitive
   ENDIF
   IF N_Elements(fieldfont) GT 0 THEN BEGIN
      self->SetWidgetFonts, widget_fieldfont=fieldfont
   ENDIF

   IF N_Elements(min_valid) GT 0 THEN BEGIN
      IF String(min_valid[0]) NE '' THEN $
         self.min_valid = Ptr_New(min_valid[0]) $
      ELSE $
         self.min_valid = Ptr_New(/Allocate_Heap)
   ENDIF 

   IF N_Elements(max_valid) GT 0 THEN BEGIN
      IF String(max_valid[0]) NE '' THEN $
         self.max_valid = Ptr_New(max_valid[0]) $
      ELSE $
         self.max_valid = Ptr_New(/Allocate_Heap)
   ENDIF 

END


; -----------------------------------------------------------------------------
; Cleanup:
;    This method makes sure there are not pointers left on the heap.

PRO MGS_Field::Cleanup

   Ptr_Free, self.undefined
   Ptr_Free, self.min_valid
   Ptr_Free, self.max_valid

   self->MGS_BaseGUI::Cleanup

END 


; -----------------------------------------------------------------------------
; Init:
;    This method initializes the text input field  

; **** NOTE: xsize or scr_size currently lost in Nirwana ! ****

FUNCTION MGS_Field::Init, $ 
   Column=column, $                 ; Set this keyword to have Label above Text Widget.
   CR_Only=cr_only, $               ; Set this keyword if you only want Carriage Return events.
   Decimal=decimal, $               ; Set this keyword to the number of digits to the right of the decimal point in FLOAT.
   Digits=digits, $                 ; Set this keyword to the number of digits permitted in INTEGER values.
   Min_Valid=min_valid, $     ; minimum valid number
   Max_Valid=max_valid, $     ; maximum valid number
   FieldFont=fieldfont, $           ; The font name for the text in the Text Widget.
   LabelText=labeltext, $           ; The text for the input field label
   LabelSize=labelsize, $           ; The X screen size of the Label Widget.
   NoEdit=noedit, $                 ; Setting this keywords makes the text widget non-editable.
   NonSensitive=nonsensitive, $     ; Setting this keywords makes the text widget non-sensitive.
   Positive=positive, $             ; Set this keyword to indicate only positive numbers allowed in the field.
   Allow_NaN=allow_nan, $           ; Show 'NaN' in the text widget if number is not finite
   Regex=regex,  $                  ; A regular expression that the input string must fulfill
   Character_Mask=character_mask, $ ; A list of valid characters
   Fold_case=fold_case, $           ; Ignore case for string validation
   Scr_XSize=scr_xsize, $           ; The X screen size of the text widget.
   Scr_YSize=scr_ysize, $           ; The Y screen size of the text widget.
   Undefined=undefined, $           ; Set to the value for "undefined" field values.
   Value=value, $                   ; The "value" of the compound widget.
   XSize=xsize, $                   ; The X size of the Text Widget.
   Frame=frame, $                   ; Set this keyword to draw a frame around the label and field
   _Extra=extra


   ;; Initialize parent object
   IF not self->MGS_BaseGUI::Init(_Extra=extra) THEN RETURN, 0
   
   ;; Error Handling.
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error initializing object'
      RETURN, 0
   ENDIF

   ;; Check keyword values.

   IF N_Elements(column) EQ 0 THEN self.column_layout = 0

   IF N_Elements(digits) EQ 0 THEN digits = 0 ELSE digits = Fix(digits)
   IF N_Elements(decimal) EQ 0 THEN decimal = -1 ELSE decimal = Fix(decimal)
   IF N_Elements(regex) EQ 0 THEN regex = ''
   IF N_Elements(character_mask) EQ 0 THEN character_mask = ''

   IF N_Elements(fieldfont) EQ 0 THEN fieldfont = self.defaultfont
   IF N_Elements(labelsize) EQ 0 THEN labelsize = 0
   IF N_Elements(labeltext) EQ 0 THEN labeltext = ''
   IF N_Elements(scr_xsize) EQ 0 THEN scr_xsize = 0
   IF N_Elements(scr_ysize) EQ 0 THEN scr_ysize = 0
   IF N_Elements(value) EQ 0 THEN value = ""
   IF N_Elements(undefined) EQ 0 THEN undefined = 'NO_VALID_DATA'
   IF N_Elements(xsize) EQ 0 THEN xsize = 0

   ;; Test regular expression if provided
   IF regex NE '' THEN BEGIN
      IF NOT self->TestRegEx(regex) THEN RETURN, 0
   ENDIF

   ;; Populate the object.

   self.cr_only = Keyword_Set(cr_only)
   self.decimal = decimal
   self.digits = digits
   self.positive = Keyword_Set(positive)
   self.allow_nan = Keyword_Set(allow_nan)

   self.regex = regex
   self.character_mask = character_mask
   self.fold_case = Keyword_Set(fold_case)

   self.undefined = Ptr_New(undefined)
   self.noedit = Keyword_Set(noedit)
   self.nonsensitive = Keyword_Set(nonsensitive)
   self.xsize = xsize

   self.labeltext = StrTrim(labeltext,2)
   self.labelsize = labelsize
   self.fieldfont = fieldfont
   self.layout_frame = Keyword_Set(frame)   ;; inherited property (changed default)

;   self.theValue = Ptr_New(/Allocate_Heap)
   IF N_Elements(min_valid) GT 0 THEN $
      self.min_valid = Ptr_New(min_valid[0]) $
   ELSE $
      self.min_valid = Ptr_New(/Allocate_Heap)

   IF N_Elements(max_valid) GT 0 THEN $
      self.max_valid = Ptr_New(max_valid[0]) $
   ELSE $
      self.max_valid = Ptr_New(/Allocate_Heap)

   ;; Set the input value.
   self->SetValue, value

   ;; Store the value as initial value for reset
   self.oldvalue = Ptr_New(value)

   RETURN, 1
END 


; -----------------------------------------------------------------------------
; MGS_Field__Define:
;   This is the object definition for the input field object.
; It inherits from MGS_BaseGUI which provides the generic widget
; functionality, and from MGS_BaseObject the abilities to set and
; query an object name and a uvalue. The base object also provides a
; general method for display of error messages which can be directed
; to a message dialog or to the log screen via the no_dialog flag.

PRO MGS_Field__Define

   struct = { MGS_FIELD, $      ; The object class name.
              labelID: 0L, $    ; The label widget ID.
              textID: 0L, $     ; The text widget ID.
              theText: "", $    ; The actual text in the text widget before last input
              fieldfont: '', $  ; The font name for the text widget
              xsize: 0L, $      ; The size of the text widget
              labeltext: '', $  ; The text of the label
              labelsize: 0L, $  ; The screen size of the label
              cr_only: 0L, $    ; A flag meaning send only carriage return events.
              tabnext: 0L, $    ; The identifier of a widget to receive the cursor focus if a TAB character is detected.
              decimal: 0, $     ; The number of decimals points in FLOAT and DOUBLE numbers.
              digits: 0, $      ; The number of digits in INT and LONG numbers.
              positive: 0, $    ; A flag meaning only positive numbers allowed.
              allow_nan: 0, $   ; Show 'NaN' in the text widget if number is not finite
              min_valid: Ptr_New(), $
              max_valid: Ptr_New(), $
              regex: "", $      ; A regular expression that the input string must fulfill
              character_mask: "", $ ; A set of valid characters 
              fold_case: 0, $   ; Validate string data ignoring the case 
              datatype: "",$    ; The type of data to be returned from the text widget.
              gentype: "", $    ; The "general" type of data: INTEGER, UNSIGNED, FLOAT, or STRING.
              undefined: Ptr_New(), $ ; The "undefined" value. Used in Get_Value methods, etc.
              noedit: 0L, $     ; A flag indicating whether text widget is editable (0) or not (1).
              nonsensitive: 0L, $ ; A flag indicating whether text widget is sensitive (0) or not (1).

              inherits MGS_BaseGUI  }

END 





; -----------------------------------------------------------------------------
; Example:
;   Test program for the field object widget. Two widgets are created
; and managed: one as a blocking, the other one as a non-blocking
; widget. You can try out other keywords from the Init method of
; MGS_Field or MGS_BaseGUI, and you can optionally return the object
; reference to the non-blocking widget. If object is not a named
; variable, the object will be automatically destroyed when you kill
; the widget, otherwise you have to delete the object yourself.

PRO Example, object=object,  _Extra=extra


   ;; Find out if autodestroy must be activated
   IF Arg_Present(object) EQ 0 THEN Destroy_Upon_Cleanup = 1

   ;; Create two field objects: one numeric, one string type
   ;; The numeric widget will be non-blocking and has predefined
   ;; buttons specifically set, the string widget has buttons
   ;; generated automatically.

   f1 = Obj_New('MGS_Field', value=!DPI, labeltext='Numeric value:', $
                Min_Valid=0., Max_Valid=1000., _Extra=extra, $
                Destroy_Upon_Cleanup=Destroy_Upon_Cleanup, $
                Buttons=['Accept', 'Cancel', 'Reset'] )

   IF NOT Obj_Valid(f1) THEN BEGIN
      Message, 'Unable to initialise first example object!'
   ENDIF 

   f2 = Obj_New('MGS_Field', value='Martin Schultz', labeltext='String value:', $
                 _Extra=extra)

   IF NOT Obj_Valid(f2) THEN BEGIN
      Obj_Destroy, f1
      Message, 'Unable to initialise second example object!'
   ENDIF 

   ;; Display the two widget dialogs
   f1->GUI, y=20
   f2->GUI, /block

   ;; (this program will only continue after you exit the second
   ;; widget)
   
   ;; Retrieve the values of both widgets
   IF Obj_Valid(f1) THEN BEGIN
      print,'Value of numeric widget: ',f1->GetValue()
   ENDIF ELSE BEGIN
      print,'Numeric widget already closed and object deleted.'
   ENDELSE 

   print,'Value of string widget: ',f2->GetValue()

   ;; Kill string object
   Obj_Destroy, f2

   ;; Return if numeric widget was already closed
   IF NOT Obj_Valid(f1) THEN RETURN

   ;; Set a different value for the numeric widget
   print,'Value of numeric widget will change in 2 seconds...'
   wait, 2
   f1->SetValue, 9.99

END
