;+
; NAME:
;    MGS_MapSetup
;
; PURPOSE:
;    This object provides a GUI interface for setting up a map
;  projection. It provides a preview window with which the user can
;  see how his/her settings will take effect and allows to control the
;  basic map properties. The object inherits from the generic
;  MGS_BaseGUI and demonstrates several features of the MGS_BaseGUI
;  class hierarchy. *** Presently, not all "compound" widgets are class
;  members: the draw widget itself, the droplist, and the checkboxes
;  should become object widgets, too, at some point. ***
;
; CATEGORY:
;  Widget Objects
;
; CALLING SEQUENCE:
;    theMap = Obj_New('MGS_MapSetup' [, keywords])
;    theMap->GUI [,/block]
;    result = theMap->GetState(ok=ok [,keywords])
;    ; result will then contain the keywords needed for map_set.
;    ; *** PLANNED TO RETURN E_Continents and E_GRID structures ***
;
; ARGUMENTS:
;
; KEYWORDS:
;
;    inherited keywords:
;    widget_title (string, input) -> The title of the top level
;          base. This string is displayed as window title AND at the
;          top of the dialog window.
;
;    widget_defaultfont (string, input) -> A font name to be used for
;          the text fields in the widget. Note: Font names are
;          platform dependent!
;
;    widget_labelfont (string, input) -> A font name to be used for
;          the title labels of widget sections. Note: Font names are
;          platform dependent!
;
;    no_dialog (boolean, input) -> Display error messages as text
;          rather than in a dialog box. 
;
; REQUIREMENTS:
;    Inherits from mgs_guiobject and mgs_baseobject 
;
; DESCRIPTION:
;
; MODIFICATION HISTORY:
;    mgs, 27 Mar 2001: Version 0.2
;    mgs, 28 Apr 2001: Version 0.4
;         - now uses new BaseGUI object and MGS_DrawColors compound
;           object
;         - has basic menu capabilities built in 
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################

; =============================================================================
; Object methods
; =============================================================================

; --- Set... methods: These methods are called from the generic Event
;     handler. See the BuildGUI method to learn how they are activated
;     by storing them as method tag in the widget's UValue field. Note
;     that a Set... method is not required for object widgets where
;     the value can always be queried from the object directly.
; ---

; -----------------------------------------------------------------------------
; SetIsotropic: (private)
;    This method sets the new value for the isotropic property

FUNCTION MGS_MapSetup::SetIsotropic, event

   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving checkbox property isotropic'
      RETURN, 0
   ENDIF

   self.isotropic = event.select
   
   ;; Update preview window
   self->Show

   RETURN, 0
END


; -----------------------------------------------------------------------------
; SetFilledContinents: (private)
;    This method sets the new value for the filled continents property

FUNCTION MGS_MapSetup::SetFilledContinents, event

   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving checkbox property filled continents'
      RETURN, 0
   ENDIF

   self.filled_continents = event.select
   
   ;; Update preview window
   self->Show

   RETURN, 0
END


; -----------------------------------------------------------------------------
; SetProjection: (private)
;    This method sets the new projection type upon selection

FUNCTION MGS_MapSetup::SetProjection, event

   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving projection type'
      RETURN, 0
   ENDIF

   self.projection = *event.selection

   ;; Update preview window
   self->Show

   RETURN, 0
END 


; -----------------------------------------------------------------------------
; GetLimit:
;    This method returns the current limit settings

FUNCTION MGS_MapSetup::GetLimit

   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving map limits'
      RETURN, 0
   ENDIF

   fieldObj = self.compound->Get(Name='LatRangeField')
   IF Obj_Valid(fieldObj) THEN BEGIN
      llat = fieldObj->GetValue()
      IF Max(ABS(llat)) LT 1.e-3 THEN llat = [ -90., 90. ]
   ENDIF ELSE BEGIN
      llat = [ -90., 90. ]
   ENDELSE 
   fieldObj = self.compound->Get(Name='LonRangeField')
   IF Obj_Valid(fieldObj) THEN BEGIN
      llon = fieldObj->GetValue()
      IF Max(ABS(llon)) LT 1.e-3 THEN llon = [ 0., 360. ]
   ENDIF ELSE BEGIN
      llon = [ 0., 360. ]
   ENDELSE 
   limit = [ llat[0], llon[0], llat[1], llon[1] ]

   RETURN, limit
END 


; -----------------------------------------------------------------------------
; GetState:
;    This method constructs a structure containing fields with values
; set to the current state of the object.
;    This method overwrites the generic GetState method (and does not
; even call it).
;    The ok keyword can be used to return information about the validity
; of the current object state.
;    As background is not an allowed keyword for map_set, it must be
; retrieved independently and used with an Erase command.

FUNCTION MGS_MapSetup::GetState, ok=ok, $
                     clon=clon, $
                     clat=clat, $
                     backgroundcolor=backgroundcolor, $
                     continentcolor=continentcolor, $
                     gridcolor=gridcolor, $
                     _Ref_Extra=e

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving object state'
      RETURN, 0
   ENDIF

   ;; Make sure the structure reflects the current object state
   self->UpdateObject

   fieldObj = self.compound->Get(Name='CLonField')
   IF Obj_Valid(fieldObj) THEN BEGIN
      clon = fieldObj->GetValue()
   ENDIF ELSE BEGIN
      clon = 0.
   ENDELSE 
   fieldObj = self.compound->Get(Name='CLatField')
   IF Obj_Valid(fieldObj) THEN BEGIN
      clat = fieldObj->GetValue()
   ENDIF ELSE BEGIN
      clat = 0.
   ENDELSE 

   limit = self->GetLimit()

   colorObj = self.compound->Get(Name='Colors')
   IF Obj_Valid(colorObj) THEN BEGIN
      colors = colorObj->GetValue()
   ENDIF ELSE BEGIN
      self->ErrorMessage, 'Color object not valid'
      stop
   ENDELSE 

   continentcolor  = colors[0]
   gridcolor       = colors[1]
   backgroundcolor = colors[2]

   struct = Create_Struct( $
                            self.projection, 1, $
                            'isotropic', self.isotropic,  $
                            'noborder', 1,  $
                            'limit', limit )


   RETURN, struct
END


; -----------------------------------------------------------------------------
; UpdateObject:  (private)
;    This method simply gathers all the information in the graphical
; user interface (that doesn't update itself) and updates the self
; object. If fields are undefined, they are set to their default values.
;   This method demonstrates how to access information from compound
; object widgets.

PRO MGS_MapSetup::UpdateObject

   ;; Call inherited method (does nothing here, but for the sake of
   ;; demonstration)
   self->MGS_BaseGUI::UpdateObject

END


; -----------------------------------------------------------------------------
; ResetDialog:
;    This method reacts on the Reset button if defined in the
; GUI. Most of the action is handled in the BaseGUI object already,
; but we need to redraw here as well.

FUNCTION MGS_MapSetup::ResetDialog, event

   void = self->MGS_BaseGUI::ResetDialog(event)

   self->Show

END


; -----------------------------------------------------------------------------
; ReDraw:
;    This method redraws the widget if an event is received from a
; compound widget.

FUNCTION MGS_MapSetup::ReDraw, event

   help,event,/stru    ;; *** only for demonstration ***

   self->Show

   RETURN, 0
END


; -----------------------------------------------------------------------------
; Show:  (private)
;    This method updates the draw widget preview of the map settings

PRO MGS_MapSetup::Show

   IF NOT Widget_Info(self.tlb, /Valid_ID) THEN RETURN

   ;; Call inherited Show method to update compound widget display
   self->MGS_BaseGUI::Show

   ;; Get ID of current plot window
   currentwindow = !D.Window

   ;; Get window ID of draw widget
   Widget_Control, self.drawid, Get_Value=wid
   Wset, wid

   ;; Define map position vector
   position = [0,0,1,1]

   keywords = self->GetState(backgroundcolor=backgroundcolor, $
                             continentcolor=continentcolor, $
                             gridcolor=gridcolor, $
                             clon=clon, $
                             clat=clat)

   IF self.debug GT 1 THEN help, keywords, /stru
   ;; Set up map with current settings in current window and display
   ;; continent outlines 

   erase, color=FSC_Color(backgroundcolor, !P.Background)
   map_set, clat, clon, _Extra=keywords, /NoErase

   map_continents, color=FSC_Color(continentcolor, !P.Color), $
      fill=(self.filled_continents GT 0)

   ;; Information only available to this object (up to now):
   ;; grid color, filed_continents
   map_grid, color=FSC_Color(gridcolor, !P.Color)

   ;; Reset window ID to window that was active before
   IF currentwindow NE -1 THEN WSet, currentwindow

END


; -----------------------------------------------------------------------------
; ProjectionNames:  (private)
;    This method defines the short and long version of the dropdown
; list for projection types. You may adapt the short list according to
; your preferences.

FUNCTION MGS_MapSetup::ProjectionNames, shortlist=shortlist

   IF Keyword_Set(shortlist) THEN BEGIN
      RETURN, [ 'Cylindrical',  $
                'Goodeshomolosine',  $
                'Mercator',  $
                'Orthographic',   $
                'Stereographic' ]
   ENDIF ELSE BEGIN
      RETURN, [ 'Aitoff',  $
                'Albers',  $
                'Azimuthal',   $
                'Conic',   $
                'Cylindrical',  $
                'Gnomic',   $
                'Goodeshomolosine',  $
                'Hammer',  $
                'Lambert',   $
                'Mercator',  $
                'Miller',   $
                'Mollweide',   $
                'Orthographic',   $
                'Robinson',  $
                'Satellite',   $
                'Sinusoidal',  $
                'Stereographic', $
                'Transverse_Mercator' ]
   ENDELSE

END


; -----------------------------------------------------------------------------
; BuildGUI:  (private)
; This method builds the graphical user interface. Here, just a simple
; compound widget for text input is added to the empty frame provided
; by the generic BaseGUI Object.

PRO MGS_MapSetup::BuildGUI

   ;; Build Menu if asked for
   IF self.menuID GT 0 THEN BEGIN
      fileID = Widget_Button(self.menuID, value='File')
      resetid = Widget_Button(fileID, value='Reset', $
                              UValue={method:'ResetDialog', object:self})
      quitid = Widget_Button(fileID, value='Quit', $
                              UValue={method:'CloseDialog', object:self})
   ENDIF 


   ;; Arrange the following widgets in one row (not actually needed)
   row1base = Widget_Base(self.layoutID, Row=1)

   ;; left column of widget: Draw widget for preview and projection
   ;; type (droplist)
   leftcol = Widget_Base(row1base, Col=1, Frame=0)
   self.drawid = Widget_Draw(leftcol, xsize=250, YSize=150)
   droprow = Widget_Base(leftcol, Row=1, Base_Align_Center=1)
   all_projections = self->ProjectionNames(shortlist=self.use_short_list)
   projectionIndex = (Where(all_projections EQ self.projection))[0] > 0
   projectionID = FSC_Droplist(droprow, Value=all_projections, Index=projectionIndex, $
      UValue={method:'SetProjection', object:self}, Title='Projection Type:', Space=[1,1])

   ;; right column: color selection, isotropic option
   ;; (*** others to follow ***)
   rightcol = Widget_Base(row1base, Col=1, Frame=0)

   ;; Color selection
   framebase = Widget_Base(rightcol, Col=1, frame=1)

   ;; colorbase is the base ID for the drawcolor compound widgets
   colorbase = Widget_Base(framebase, Row=1)

   ;; Coordinates (Center coordinates and Limit)
   framebase = Widget_Base(rightcol, Col=1, frame=1)
   coordlabel = Widget_Label(framebase, Value='Center Coordinates', $
                             font=self.labelfont, /Align_Left)

   coordbase = Widget_Base(framebase, Col=1)

   ;; Options: isotropic, filled continents
   framebase = Widget_Base(rightcol, Col=1, frame=1)
   optionslabel = Widget_Label(framebase, Value='Options', $
                             font=self.labelfont, /Align_Left)

   base = Widget_Base(framebase, /Nonexclusive)
   isotropicID = Widget_Button(base, Value='isotropic', $
      UValue={method:'SetIsotropic', object:self})
   IF self.isotropic THEN Widget_Control, isotropicID, /Set_Button

   filledcontID = Widget_Button(base, Value='filled continents', $
      UValue={method:'SetFilledContinents', object:self})
   IF self.filled_continents THEN Widget_Control, filledcontID, /Set_Button


   ;; Save colorbase and coordbase for addition of compound widgets
   self.cwbaseid = Ptr_New( [ colorbase, coordbase] )

   RETURN

END


; -----------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user. Normally, the user should use the GetState() method to
; retrieve the object state in a usable form.

PRO MGS_MapSetup::GetProperty, $
   clon=clon,    $            ; The center longitude coordinate
   clat=clat,    $            ; The center latitude coordinate
   _Ref_Extra=extra           ; Inherited and future keywords
                              ;
                              ; Inherited keywords:
                              ; name      : The variable name
                              ; uvalue    : a user-defined value
                              ; window_title
                              ; widget_title
                              ; widget_defaultfont
                              ; widget_labelfont
                              ; set_value_pro
                              ; get_value_func


   ;; Get properties from base object
   self->MGS_BaseGUI::GetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving object properties!'
      RETURN
   ENDIF

   ;; Return object properties
   IF Arg_Present(clon) THEN BEGIN
      fieldObj = self.compound->Get(Name='CLonField')
      IF Obj_Valid(fieldObj) THEN $
         clon = fieldObj->GetValue()
   ENDIF
   IF Arg_Present(clat) THEN BEGIN
      fieldObj = self.compound->Get(Name='CLatField')
      IF Obj_Valid(fieldObj) THEN $
         clat = fieldObj->GetValue()
   ENDIF

END


; -----------------------------------------------------------------------------
; SetProperty:
; This method sets specific object values.

PRO MGS_MapSetup::SetProperty, $
     clon=clon,    $ ; The center longitude coordinate
     clat=clat,    $ ; The center latitude coordinate
     _Extra=extra                  ;
                                   ; Inherited keywords:
                                   ; name      : The variable name
                                   ; no_copy   : Don't keep local copy
                                   ;             of uvalue
                                   ; no_dialog : Don't display
                                   ;             interactive dialogs
                                   ; uvalue    : a user-defined value
                                   ; window_title
                                   ; widget_title
                                   ; row_layout
                                   ; no_frame
                                   ; widget_defaultfont
                                   ; widget_labelfont
                                   ; set_value_pro
                                   ; get_value_func


   ;; Set Properties of base object
   self->MGS_BaseGUI::SetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error setting object properties'
      RETURN
   ENDIF

   ;; Replace value of center coordinates
   IF N_Elements(clon) GT 0 THEN BEGIN
      fieldObj = self.compound->Get(Name='CLonField')
      IF Obj_Valid(fieldObj) THEN $
         fieldObj->SetValue, Float(clon)
   ENDIF
   IF N_Elements(clat) GT 0 THEN BEGIN
      fieldObj = self.compound->Get(Name='CLatField')
      IF Obj_Valid(fieldObj) THEN $
         fieldObj->SetValue, Float(clat)
   ENDIF

   ;; Make sure object is up-to-date and redisplay
   self->UpdateObject
   self->Show

END


; -----------------------------------------------------------------------------
; Init:
;   This method initializes the map object. It does not build a GUI,
; but only sets the object property varioables.

FUNCTION MGS_MapSetup::Init, $
     projection=projection,  $  ; The projection type of the map
     short_list=use_short_list,  $  ; Only limited selection of projections
     color=fgcolor,    $        ; The map drawing color name
     background=bgcolor,  $     ; The name of the background color
     gridcolor=gdcolor,  $      ; The name of the grid color (default=drawing)
     isotropic=isotropic, $     ; Set isotropic projection
     filled_continents=filled_continents, $  ; Fill continents with fgcolor
     clon=clon, $               ; Center longitude (*** not implemented!***)
     clat=clat, $               ; Center latitude  (*** not implemented!***)
     _Extra=extra  ; Extra keywords from inherited objects
                                ;
                                ; Inherited keywords (from
                                ; MGS_BaseGUI and MGS_BaseObject):
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value
                                ; window_title
                                ; widget_title
                                ; row_layout
                                ; no_frame
                                ; widget_defaultfont
                                ; widget_labelfont
                                ; set_value_pro
                                ; get_value_func


   ;; Initialize parent object first
   IF not self->MGS_BaseGUI::Init(_Extra=extra) THEN RETURN, 0

   ;; Error Handler 
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error initializing object'
      RETURN, 0
   ENDIF

   ;; Check for keywords and parameters.

   ;; Set projection (needs more error checking?)
   IF N_Elements(projection) EQ 0 THEN $
      self.projection='Cylindrical'  $
   ELSE $
      self.projection = projection

   self.use_short_list = Keyword_Set(use_short_list)

   ;; Set colors
   colorlabels = ['Continents', 'Grid', 'Background']
   colors      = ['Moccasin',   'Moccasin', 'Navy' ]

   IF N_Elements(fgcolor) GT 0 THEN colors[0] = fgcolor[0]
   IF N_Elements(gdcolor) GT 0 THEN colors[1] = gdcolor[0]
   IF N_Elements(bgcolor) GT 0 THEN colors[2] = bgcolor[0]

   colorObj = Obj_New('MGS_DrawColors', $
                      name='Colors', $
                      colors=colors, $
                      labeltext=colorlabels, $
                      startindex=!D.Table_Size-4, $
                      title='Colors', $
                      NotifyObject={object:self, method:'Redraw'}, $
                      hook=0)

   IF Obj_Valid(colorObj) THEN self.compound->Add, colorObj

   ;; Isotropic projection?
   self.isotropic = Keyword_Set(isotropic)

   ;; Filled continents?
   self.filled_continents = Keyword_Set(filled_continents)

   IF N_Elements(clon) EQ 0 THEN clon = 0.
   IF N_Elements(clat) EQ 0 THEN clat = 0.

   limitdefault = [ 0., 0., 0., 0. ]
   limittext = [ 'min lat.', 'min lon.', 'max lat.', 'max lon.']

   fieldObj = Obj_New('MGS_Field', $
                      Name='CLonField', $
                      Value=Float(clon), $
                      NotifyObject={object:self, method:'Redraw'}, $
                      hook=1, $
                      Labeltext='Longitude: ', $
                      Label_Left=1, $
                      XSize=10, $
                      Decimal=2, $
                      Min_Valid=-180., $
                      Max_Valid=540., $
                      LabelSize=96, $
                      /Align_Left, $
                      /Cr_Only)
   IF Obj_Valid(fieldObj) THEN self.compound->Add, fieldObj


   fieldObj = Obj_New('MGS_Field', $
                      Name='CLatField', $
                      Value=Float(clat), $
                      NotifyObject={object:self, method:'Redraw'}, $
                      hook=1, $
                      Labeltext='Latitude: ', $
                      Label_Left=1, $
                      XSize=10, $
                      Decimal=2, $
                      Min_Valid=-90., $
                      Max_Valid=90., $
                      LabelSize=96, $
                      /Align_Left, $
                      /Cr_Only)
   IF Obj_Valid(fieldObj) THEN self.compound->Add, fieldObj

   fieldObj = Obj_New('MGS_RangeField', $
                      Name='LatRangeField', $
                      Value=Float([limitdefault[0],limitdefault[2]]), $
                      NotifyObject={object:self, method:'Redraw'}, $
                      hook=1, $
                      Labeltext=['Lat. min:','max:'], $
                      Title='Latitude Range', $
                      /No_Frame, $
                      Label_Left=1, $
                      XSize=10, $
                      Decimal=2, $
                      Min_Valid=-90., $
                      Max_Valid=90., $
;;                      LabelSize=96, $
                      /Align_Left, $
                      /Cr_Only)

   IF Obj_Valid(fieldObj) THEN self.compound->Add, fieldObj

   fieldObj = Obj_New('MGS_RangeField', $
                      Name='LonRangeField', $
                      Value=Float([limitdefault[1],limitdefault[3]]), $
                      NotifyObject={object:self, method:'Redraw'}, $
                      hook=1, $
                      Labeltext=['Lon. min:','max:'], $
                      Title='Longitude Range', $
                      /No_Frame, $
                      Label_Left=1, $
                      XSize=10, $
                      Decimal=2, $
                      Min_Valid=-180., $
                      Max_Valid=540., $
                      /Allow_Wrap, $        ;; lon0 may be greater than lon1
;;                      LabelSize=96, $
                      /Align_Left, $
                      /Cr_Only)

   IF Obj_Valid(fieldObj) THEN self.compound->Add, fieldObj

   IF self.window_title EQ 'Generic widget object' THEN $
      self.window_title = 'Map properties'

   RETURN, 1
END


; -----------------------------------------------------------------------------
; MGS_MapSetup__Define:
; This is the object definition for the map configuration tool.

PRO MGS_MapSetup__Define

   struct = { MGS_MapSetup, $  ; The MGS_MAP object class.
              projection:'', $  ; The current map projection
              use_short_list:0,  $ ; Use short selection list for projection
              isotropic: 0, $   ; Display the map isotropically
              filled_continents: 0,  $ ; Fill continents with fgcolor

              ;; Widget identifiers
              drawid: 0L,   $              ; The widget ID of the preview window

              inherits MGS_BaseGUI  $ ; provides basic general properties
            }
END



; -----------------------------------------------------------------------------
; Example:
;   Wrapper for the MapSetup dialog or application. Try out:
; (1) example, object=o   ; need to destroy o yourself later!
; (2) example, object=o, /Destroy   ; widget kills object if closed
; (*** well: should!!!!! ***)
; (3) example, /block, buttons=['OK','Cancel','Reset']
; (4) example, /menu
; NOTES:
; (1) The Reset functionality only works for those widgets that are
; derived from MGS_BaseGUI. Hence, neither the projection type nor the
; checkboxes are reset automatically!
; (2) The /Destroy_Upon_Cleanup switch apparently destroys all
; compound widgets of MGS_MapSetup but not the object itself.

PRO Example, block=block, object=object, _Extra=extra

   themap = Obj_New('MGS_MapSetup',_Extra=extra)
   themap->GUI, block=keyword_Set(block)

   IF Obj_Valid(themap) THEN BEGIN
      keywords = themap->GetState()
      help, keywords, /structure
      IF Arg_Present(object) EQ 0 THEN Obj_Destroy, themap ELSE object=themap
   ENDIF 

END
