;+
; NAME:
;   mgs_variable (object)
;
; PURPOSE:
;     This object is a model for a scientific data variable. It
;   derives many basic properties from the MGS_BaseVariable object
;   and enhances its functionality through the addition of a few
;   standard data properties (long_name, units, missing_value,
;   min_valid, and max_valid). The numerical filter is extended with
;   the non_missing and valid_only options. Also, a few numerical
;   tools have been added (Mean, Median, Min, Max). While this object
;   is primarily designed for numerical variables, it can still be
;   used for variables of other types. Note, however, that calling a
;   numerical routine with such data will cause an error message. 
;
;   See the description of MGS_BaseVariable for other features of this
;   object.
;
; AUTHOR:
;
;   Dr. Martin Schultz
;   Max-Planck-Institut fuer Meteorologie
;   Bundesstr. 55, D-20146 Hamburg
;   email: martin.schultz@dkrz.de
;
; CATEGORY:
;   General object programming
;
; CALLING SEQUENCE:
;
;
;
; EXAMPLE:
;   See example procedure at the bottom of this file
;
; MODIFICATION HISTORY:
;   Martin G. Schultz, 30 May 2000: VERSION 1.00
;   mgs, 11 Mar 2001: VERSION 1.10 
;       - newly derived from older version of MGS_BaseVariable
;       - clearer distinction between basevariable and (scientific)
;         variable objects
;       - added missing_value, min_valid, and max_valid properties
;   mgs, 29 May 2001: 
;       - adapted to new GUI object class (uses mgs_inputmask)
;       - improved handling of data change in setproperty
;   mgs, 03 Jun 2001:
;       - bug warning in CallFunction, a few comments changed
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2000-2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################


; =====================================================================================
; Utility methods:
;     Mean         : Compute the mean of the data stored in the object
;     Median       : Compute the median of the data stored in the object
;     Min          : Compute the minimum data value
;     Max          : Compute the maximum data value
;     CallFunction : Generic interface for function calls
;     Scale        : apply scaling factor and offset to data values

; -------------------------------------------------------------------------------------
; Mean:
;   This method computes the mean of the variable. The dimension
; argument allows to compute the mean for all elements of this dimension,
; i.e. in this case mean returns a vector rather than a scalar. See
; the CallFunction method for further explanation.


function MGS_Variable::Mean,  $
               data,          $            ; [optional] Use external data
               dimension=dimension,     $  ; compute mean over this dimension
               min_substitute=minsubs,  $  ; a substitute value for data < min_valid
               max_substitute=maxsubs,  $  ; a substitute value for data > max_valid
               result=result, $            ; a status value 
               nvalid=nvalid               ; the number of valid data values 

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   retval = !Values.F_NaN
   result = 'NO_VALID_DATA'
   nvalid = 0L

   argok = self->UseDataArgument(data, arg_present(data)) 
 
   retval = self->CallFunction(data, $
                               dimension=dimension,     $
                               min_substitute=minsubs,  $
                               max_substitute=maxsubs,  $
                               result=result,           $
                               argok=argok,             $
                               nvalid=nvalid,           $
                               name='Mean' )

   RETURN, retval

END


; -------------------------------------------------------------------------------------
; Median:
;   This method computes the median of the variable. The dimension
; argument allows to compute the mean for all elements of this dimension,
; i.e. in this case mean returns a vector rather than a scalar. See
; the CallFunction method for further explanation.


function MGS_Variable::Median,  $
               data,          $            ; [optional] Use external data
               dimension=dimension,     $  ; compute mean over this dimension
               min_substitute=minsubs,  $  ; a substitute value for data < min_valid
               max_substitute=maxsubs,  $  ; a substitute value for data > max_valid
               result=result, $            ; a status value 
               nvalid=nvalid               ; the number of valid data values 

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   retval = !Values.F_NaN
   result = 'NO_VALID_DATA'
   nvalid = 0L

   argok = self->UseDataArgument(data, arg_present(data)) 
 
   retval = self->CallFunction(data, $
                               dimension=dimension,     $
                               min_substitute=minsubs,  $
                               max_substitute=maxsubs,  $
                               result=result,           $
                               argok=argok,             $
                               nvalid=nvalid,           $
                               name='Median' )

   RETURN, retval

END


; -------------------------------------------------------------------------------------
; Min:
;   This method computes the minimum of the variable. The dimension
; argument allows to compute the mean for all elements of this dimension,
; i.e. in this case mean returns a vector rather than a scalar. See
; the CallFunction method for further explanation.


function MGS_Variable::Min,  $
               data,          $            ; [optional] Use external data
               dimension=dimension,     $  ; compute mean over this dimension
               min_substitute=minsubs,  $  ; a substitute value for data < min_valid
               max_substitute=maxsubs,  $  ; a substitute value for data > max_valid
               result=result, $            ; a status value 
               nvalid=nvalid               ; the number of valid data values 

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   retval = !Values.F_NaN
   result = 'NO_VALID_DATA'
   nvalid = 0L

   argok = self->UseDataArgument(data, arg_present(data)) 
 
   retval = self->CallFunction(data, $
                               dimension=dimension,     $
                               min_substitute=minsubs,  $
                               max_substitute=maxsubs,  $
                               result=result,           $
                               argok=argok,             $
                               nvalid=nvalid,           $
                               name='Min' )

   RETURN, retval

END


; -------------------------------------------------------------------------------------
; Max:
;   This method computes the maximum of the variable. The dimension
; argument allows to compute the mean for all elements of this dimension,
; i.e. in this case mean returns a vector rather than a scalar. See
; the CallFunction method for further explanation.


function MGS_Variable::Max,  $
               data,          $            ; [optional] Use external data
               dimension=dimension,     $  ; compute mean over this dimension
               min_substitute=minsubs,  $  ; a substitute value for data < min_valid
               max_substitute=maxsubs,  $  ; a substitute value for data > max_valid
               result=result, $            ; a status value 
               nvalid=nvalid               ; the number of valid data values 

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   retval = !Values.F_NaN
   result = 'NO_VALID_DATA'
   nvalid = 0L

   argok = self->UseDataArgument(data, arg_present(data)) 
 
   retval = self->CallFunction(data, $
                               dimension=dimension,     $
                               min_substitute=minsubs,  $
                               max_substitute=maxsubs,  $
                               result=result,           $
                               argok=argok,             $
                               nvalid=nvalid,           $
                               name='Max' )

   RETURN, retval

END


; -------------------------------------------------------------------------------------
; CallFunction:
;   This method extracts the valid data from the object or the given
; argument and calls the function given in the functionname string
; keyword. By default, data are filtered with the valid_only flag,
; i.e. the function is evaluated only for finite, non-missing data
; between min_valid and max_valid. If you specify a substitute value
; for data < min_valid or for data > max_valid (e.g. 0. or 1/2
; detection limit), then data outside the range are temporarily
; replaced with the substitute before the function call. 
;
;   If a dimension is given with the dimension keyword, the method
; will loop over this dimension and call the function for each element
; along this dimension. This currently only works for data without
; missing or infinite values. 
;
;   Use the nvalid keyword to return the number of valid data values
; used in the computation.

; ***** DIMENSION keyword should evaluate function across given
; dimension rather than for all values of this dimension. Otherwise
; it would be hard to compute e.g. zonal mean values for individual
; time steps!!!

FUNCTION MGS_Variable::CallFunction,  $
               data,          $            ; [optional] Use external data
               dimension=dimension,     $  ; compute mean over this dimension
               min_substitute=minsubs,  $  ; a substitute value for data < min_valid
               max_substitute=maxsubs,  $  ; a substitute value for data > max_valid
               result=result,           $  ; a status value 
               argok=argok,             $  ; use external data argument (private)
               nvalid=nvalid,           $  ; the number of valid data values 
               name=name,               $  ; the name of the function to be called
               _Extra=extra


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   retval = !Values.F_NaN
   IF N_Elements(result) EQ 0 THEN result = 'NO_VALID_DATA'
   nvalid = 0L
 
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error computing '+name+' of '+self.name
      RETURN, retval
   ENDIF

   ;; Determine whether to use object's data
   IF N_Elements(argok) EQ 0 THEN  $
      argok = self->UseDataArgument(data, arg_present(data)) 
   IF argok EQ 0 THEN BEGIN
      data = self->GetData(result=result)
   ENDIF ELSE IF argok EQ 1 THEN BEGIN
      result = 'OK'
   ENDIF 

   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      self->ErrorMessage, 'No data to convert!', /Warning
      RETURN, retval
   ENDIF

   ;; If data are not numeric, return with error
   IF NOT self->IsNumeric(data, isdouble=isdouble) THEN BEGIN
      result = 'INVALID_TYPE'
      self->ErrorMessage, 'Invalid data type! Variable '+self.name
      RETURN, retval
   ENDIF

   ;; Determine if range check needed later
   IF N_Elements(minsubs) GT 0 AND Finite(*self.min_valid) THEN checkmin = 1
   IF N_Elements(maxsubs) GT 0 AND Finite(*self.max_valid) THEN checkmax = 1

   ;; Get indices of valid data values. If a substitute value for
   ;; data < min_valid or data > max_valid is provided, get indices of
   ;; non_missing data. The range filtering is then done below.
   IF Keyword_Set(checkmin) OR Keyword_Set(checkmax) THEN BEGIN
      wok = self->Filter(data, /non_missing)
   ENDIF ELSE BEGIN
      wok = self->Filter(data, /valid_only)
   ENDELSE

   ;; If dimension argument is passed, evaluate mean across that
   ;; dimension
   ;; *** NOT FULLY IMPLEMENTED ! NO TEST FOR MISSING VALUE ETC. ***

   IF N_Elements(dimension) EQ 1 THEN BEGIN
      IF N_Elements(wok) NE N_Elements(data) THEN BEGIN
         self->ErrorMessage, 'Sorry! '+name+' along dimension operates only on valid data!'
         RETURN, retval
      ENDIF

      thisdim = Fix(dimension) 
      
      ;; Make sure dimensions are in range
      ndims = Size(data, /N_Dimensions)
      dims  = Size(data, /Dimensions)
      IF thisdim LT 0 OR thisdim GT ndims THEN BEGIN
         self->ErrorMessage, 'Invalid dimension '+StrTrim(thisdim,2)+ $
            '(Max = '+StrTrim(ndims,2)+')'
         RETURN, result
      ENDIF

      ;; Create result array
      retval = dblarr(dims[thisdim-1])
      nvalid = lonarr(dims[thisdim-1])
      
      ;; Compute the respective mean values
      FOR i=0L, dims[thisdim-1]-1 DO BEGIN
         result = 'NO_VALID_DATA'
         theslice = double( self->ExtractHyperslice(data, $
                                                    dimension=thisdim, $
                                                    index=i) )
         result = 'INVALID_FUNCTION'
         IF N_Elements(extra) GT 0 THEN BEGIN
            retval[i] = Call_Function(name,theslice,_Extra=extra)
         ENDIF ELSE BEGIN
            retval[i] = Call_Function(name,theslice)
         ENDELSE
         nvalid[i] = N_Elements(theslice)
      ENDFOR
      
   ENDIF ELSE BEGIN
      ;; Make temporary copy of data only if neccessary
      ;; Always operate on double precision data
;; **** BUG?: if only min_substitute or max_substitute are
;;      provided, the other might be used as stored, but should
;;      in fact be masked out!!!
      IF Keyword_Set(checkmin) OR Keyword_Set(checkmax) THEN BEGIN
         tmpdata = double(data)
         IF Keyword_Set(checkmin) THEN BEGIN
            woor = Where(tmpdata[wok] LT *self.min_valid, cnt)
            IF cnt GT 0 THEN tmpdata[wok[woor]] = minsubs
         ENDIF
         IF Keyword_Set(checkmax) THEN BEGIN
            woor = Where(tmpdata[wok] GT *self.max_valid, cnt)
            IF cnt GT 0 THEN tmpdata[wok[woor]] = maxsubs
         ENDIF
         result = 'INVALID_FUNCTION'
         IF N_Elements(extra) GT 0 THEN BEGIN
            retval = Call_Function(name,tmpdata[wok],_Extra=extra)
         ENDIF ELSE BEGIN
            retval = Call_Function(name,tmpdata[wok])
         ENDELSE
      ENDIF ELSE BEGIN
         result = 'INVALID_FUNCTION'
         IF N_Elements(extra) GT 0 THEN BEGIN
            retval = Call_Function(name,double(data[wok]),_Extra=extra)
         ENDIF ELSE BEGIN
            retval = Call_Function(name,double(data[wok]))
         ENDELSE
      ENDELSE
      nvalid = N_Elements(wok)
   ENDELSE

   result = 'OK'

   RETURN, retval

END


; -------------------------------------------------------------------------------------
; Scale:
;    This method performs a linear operation data*factor+offset on the
; data stored in the object. The data type may change according to the
; IDL rules (e.g. scaling a LONG array with a FLOAT factor yields a
; FLOAT array). Setting the preserve_type flag will convert the final
; result back to the original type.
;   Extra keywords are passed to the ConvertType method if the
; Preserve_Type keyword is set. Useful options include Truncate,
; Round, or Clip (see ConvertType method above). Extra keywords are
; also passed to GetData so you can set other filter options in
; addition to the default finite_only.
;   Setting the units keyword allows you to change the variable's unit
; without an additional call to the SetProperty method.

PRO MGS_Variable::Scale,  $
                data,           $
                factor=factor,  $
                offset=offset,  $
                units=units,    $
                preserve_type=preserve_type,  $
                _Ref_Extra=extra


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error while scaling variable data!'
      RETURN
   ENDIF

   ;; Determine whether to use object's data
   argok = self->UseDataArgument(data, arg_present(data)) 
   IF argok EQ 0 THEN BEGIN
      data = self->GetData(result=result)
   ENDIF ELSE IF argok EQ 1 THEN BEGIN
      result = 'OK'
   ENDIF 

   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      self->ErrorMessage, 'No data to convert!', /Warning
      RETURN
   ENDIF

   ;; If data are not numeric, return with error
   IF NOT self->IsNumeric(data, isdouble=isdouble) THEN BEGIN
      result = 'INVALID_TYPE'
      self->ErrorMessage, 'Invalid data type! Variable '+self.name
      RETURN
   ENDIF

   ;; Define defaults
   tname = Size(data, /Tname)
   CASE tname OF
       'DOUBLE' : BEGIN
                    DefFactor = 1.0D
                    DefOffset = 0.0D
                 END
       'FLOAT'  : BEGIN
                    DefFactor = 1.0
                    DefOffset = 0.0
                 END
       'LONG64' : BEGIN
                    DefFactor = 1LL
                    DefOffset = 0LL
                 END
       'LONG'   : BEGIN
                    DefFactor = 1L
                    DefOffset = 0L
                 END
       'INT'    : BEGIN
                    DefFactor = 1
                    DefOffset = 0
                 END
       'ULONG64' : BEGIN
                    DefFactor = 1ULL
                    DefOffset = 0ULL
                 END
       'ULONG'  : BEGIN
                    DefFactor = 1UL
                    DefOffset = 0UL
                 END
       'UINT'   : BEGIN
                    DefFactor = 1U
                    DefOffset = 0U
                 END
       'BYTE'   : BEGIN
                    DefFactor = 1B
                    DefOffset = 0B
                 END
       
       ELSE : RETURN    ; no scaling possible
   ENDCASE
   
   ;; Keyword checking
   IF N_Elements(factor) EQ 0 THEN factor = DefFactor
   IF N_Elements(offset) EQ 0 THEN offset = DefOffset

   ;; Retrieve a local copy of the object's data and an index of all
   ;; valid data values 
   thedata = data

   ;; Convert data type if necessary (i.e. if factor or offset are of
   ;; higher rank than thedata)
   validtype = self->IsNumeric(thedata, rank=datarank) $
     * self->IsNumeric(factor, rank=factorrank) $
     * self->IsNumeric(offset, rank=offsetrank)

   IF NOT validtype THEN BEGIN
      self->ErrorMessage, 'Invalid type for factor or offset!'
      RETURN
   ENDIF

   therank = Max([ datarank, factorrank, offsetrank ])
   IF therank GT datarank THEN self->ConvertType, thedata, rank=therank

   ;; Scale valid data
   ok = self->Filter(data, /valid_only)
   thedata[ok] = offset + factor * thedata[ok]

   ;; Adjust data type name or convert data back to old type if
   ;; preserve_type keyword is set
   IF Keyword_Set(preserve_type) THEN BEGIN
       self->ConvertType, thedata, tname=self.tname, _Extra=extra
   ENDIF ELSE BEGIN
       self.tname = Size(thedata, /TName)
   ENDELSE

   ;; Store data back in object
   IF argok EQ 0 THEN BEGIN
      Ptr_Free, self.data
      self.data = Ptr_New(thedata, /No_Copy)
   
      ;; Replace unit if changed
      ;; Sideeffect: this calls the Validate method
      self->SetProperty, units=units
   ENDIF

END


; -------------------------------------------------------------------------------------
; ConvertType:
; (description copied from MGS_BaseVariable)
;   This method accepts a data argument and converts it to a specified
; data type. If no data argument is supplied, the data stored in the
; object will be used. The target data type can be given in four
; ways:
; - set the tname keyword to the name of the data type (string)
; - set one of the logical flags (double, float, ...)
; - set the integer type vale (see IDL online help on data type
;   values)
; - specify the numerical rank (numerical data only; see IsNumeric
;   method)
;
;   The truncate, round, and clip keywords determine, how data are
; converted from a "higher" to a "lower" numerical type rank. The
; truncate keyword is purely for readability of code; truncation is
; IDL's default for type conversion. Alternatively, you can round
; float data to the nearest (long64, long, or short) integer. Be aware
; that you can end up with unreasonable values in both cases, unless
; the clip keyword is set. Example: "print, fix(32768L)" yields
; -32768). Setting the clip keyword will ensure that the data does not
; extend beyond the valid data range given by the new data type. 
;
;   The minimum and maximum possible (valid) data values for the new
; type can be returned via the min_valid and max_valid keywords. In
; order to avoid machine specific differences, the limit for double
; precision values is set to 1.D300, and the limit for float is set to
; 1.E31. If you supply a named variable for the EPS keyword, the
; minimum positive value that is discernible from zero for the target
; data type will be returned. This value is machine dependent (for
; integer types, it will always be 1).
;
;   You can also use this method to convert numerical data to
; formatted strings. Use the format keyword to specify the output
; format.
;
; (added information)
;   For individual keywords, see the ConvertType method in the
; MGS_BaseVariable object.
;
;   The method is overwritten here to convert the type of the
; missing_value, min_valid, and max_valid atributes along with the
; object data.

pro MGS_Variable::ConvertType,  $
                data,                 $ ; The data to be converted
                min_valid=min_valid,  $ ; Return the minimum possible value for the new type
                max_valid=max_valid,  $ ; Return the maximum possible value for the new type
                eps=eps,  $   ; Return machine specific minimum number that is discernible from zero 
                _Extra=extra


   ;; Determine whether to use data argument or object's data:
   argok = self->UseDataArgument(data, arg_present(data))
   use_self = (argok EQ 0)

   ;; Call inherited method and apply it also to the object's missing
   ;; value, min_valid and max_valid attributes if object data are
   ;; converted. 
   IF use_self THEN BEGIN
      self->MGS_BaseVariable::ConvertType, $
         min_valid=min_valid,  $   ; The minimum allowed value for new type
         max_valid=max_valid,  $   ; The maximum allowed value for new type
         eps=eps, $               ; The minimum non-zero value
         _Extra=extra

      IF finite(*self.missing_value) THEN $
         self->MGS_BaseVariable::ConvertType, *self.missing_value, $
         _Extra=extra

      IF finite(*self.min_valid) THEN $
         self->MGS_BaseVariable::ConvertType, *self.min_valid, $
         _Extra=extra

      IF finite(*self.max_valid) THEN $
         self->MGS_BaseVariable::ConvertType, *self.max_valid, $
         _Extra=extra

   ENDIF ELSE BEGIN
      self->MGS_BaseVariable::ConvertType, data, $
         min_valid=min_valid,  $   ; The minimum allowed value for new type
         max_valid=max_valid,  $   ; The maximum allowed value for new type
         eps=eps, $
         _Extra=extra
   ENDELSE

END


; =====================================================================================
; Methods for display of variable properties and data. These involve
; special objects for GUI and data display.
;     ShowInfo : Display GUI with variable properties
;     SetInfoObject : Links an existing info object to this variable

; -----------------------------------------------------------------------------
; ShowInfo:
;   requires mgs_varinfoObject

PRO MGS_Variable::ShowInfo, block=block

   ;; If no info object has already been linked, create one now
   IF NOT Obj_Valid(self.infoObj) THEN BEGIN
;;      self.infoObj = Obj_New('MGS_VarInfoObject', self)
      self->GetProperty,  $
         name=name, $
         long_name=long_name,  $ ; A more complete description of the variable
         units=units,          $ ; The physical unit of the variable
         missing_value=missing_value, $
         min_valid=min_valid,  $
         max_valid=max_valid

      thestatus = { name:name, long_name:long_name, units:units, $
                    missing_value:missing_value, min_valid:min_valid, $
                    max_valid:max_valid }

      self.infoObj = Obj_New('MGS_InputMask', thestatus, /allow_nan, $
                             /NoEdit, /singleton, /block)
;;;;                             /destroy_upon_cleanup,debug=99)

   ENDIF

   ;; Display info about variable properties and let user change them
   self.infoObj->GUI, block=block

END


; -----------------------------------------------------------------------------
; SetInfoObject:

PRO MGS_Variable::SetInfoObject, newobject, none=none

   ;; Delete old info object if requested
   IF Keyword_Set(none) THEN Obj_Destroy, self.infoObj

   ;; If newobject is valid, delete old object and replace it
   IF Obj_Valid(newobject) THEN BEGIN
      Obj_Destroy, self.infoObj
      self.infoObj = newobject
   ENDIF

END


; =====================================================================================
; Methods for data retrieval and dimension handling:
;     GetDefaultValue (private) : return dummy value for invalid data
;     GetFilterNumMask (private) : this method enhances the inherited
;             one with additional filters

; -----------------------------------------------------------------------------
; GetDefaultValue:  (private)
;    This function returns a standard value for invalid data. This
; value is missing_value for all numerical types and '' for
; strings. The result keyword will be set to 'NO_VALID_DATA'. This
; method overwrites the GetDefaultValue method of MGS_BaseVariable
; where the numerical default value was always zero.

FUNCTION MGS_Variable::GetDefaultValue,  $
                    Result=result


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   IF self->IsNumeric(rank=rank) THEN BEGIN
      retval = self.missing_value
      result = 'NO_VALID_DATA'
   ENDIF ELSE BEGIN
   ;; Call parent's method
      retval = self->MGS_Basevariable::GetDefaultValue(result=result)
   ENDELSE

   RETURN, retval

END


; -----------------------------------------------------------------------------
; GetFilterNumMask: (private)
;   This method constructs a boolean array containing 1 for each data
; element that fulfills the combined filter criteria for numeric
; data. The inherited filter crietria finite_only, positive, negative,
; and non_zero are enhanced as follows:
;   non_missing (boolean) : returns 0 for every missing_value
;   valid_only (boolean) : data must be finite, non_missing, and
;                          within min_valid and max_valid
;   
;    No test is made if data is a valid numeric type argument. This
; must be done in the calling routine. 

FUNCTION MGS_Variable::GetFilterNumMask,  $
                   data,                    $ ; The data to be analyzed
                   non_missing=non_missing, $ ; Return only non-missing data
                   valid_only=valid_only,   $ ; Return only valid data
                   finite_only=finite_only, $ ; (inherited)
                   _Extra=extra               ; Inherited keywords:
                                 ; finite_only: Return only valid numerical data
                                 ; positive: Return only positive numerical data
                                 ; negative: Return only negative numerical data
                                 ; non_zero: Return only non-zero values


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; Default: range doesn't matter
   inrange_only = 0

   ;; Compose conditions for valid_only
   IF Keyword_Set(valid_only) THEN BEGIN
      non_missing = 1
      inrange_only = 1
   ENDIF

   IF Keyword_Set(non_missing) THEN finite_only = 1

   ;; Call inherited method
   retval = self->MGS_BaseVariable::GetFilterNumMask(data, $
                          finite_only=finite_only, $
                          _Extra=extra)

   IF Keyword_Set(non_missing) AND Finite(*self.missing_value) THEN BEGIN
      IF self->IsFloat(data, double=isdouble) THEN BEGIN
         eps = (MaChar(double=isdouble)).eps 
      ENDIF ELSE BEGIN
         eps = 0
      ENDELSE
      wmiss = Where(Abs(data-*self.missing_value) LT eps, cmiss)
      IF cmiss GT 0 THEN retval[wmiss] = 0B
   ENDIF

   IF Keyword_Set(inrange_only) THEN BEGIN
      IF Finite(*self.min_valid) THEN BEGIN
         woor = Where(data LT *self.min_valid, coor)
         IF coor GT 0 THEN retval[woor] = 0B
      ENDIF
      IF Finite(*self.max_valid) THEN BEGIN
         woor = Where(data GT *self.max_valid, coor)
         IF coor GT 0 THEN retval[woor] = 0B
      ENDIF
   ENDIF

   RETURN, retval

END


; =====================================================================================
; Standard object methods:
;     GetProperty : retrieve object values
;     SetProperty : set object values
;     Cleanup     : free object pointers and destroy
;     Init        : initialize object

; -------------------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user.

PRO MGS_Variable::GetProperty,  $
                   long_name=long_name,  $ ; A more complete description of the variable
                   units=units,          $ ; The physical unit of the variable
                   missing_value=missing_value, $
                   min_valid=min_valid,  $
                   max_valid=max_valid,  $
                   infoobject=infoobject,$
                   _Ref_Extra=extra
                                ; Inherited keywords:
                                ; data      : The data values
                                ; dim1..dim8 : The dimension "values"
                                ; dims      : The sizes of the dimensions
                                ; tname     : The data type (string)
                                ; name      : The variable name
                                ; uvalue    : a user-defined value
                                ; no_copy   : Copy pointer values?
                                ; no_dialog : Object uses dialogs for messages?
                                ; debug     : Object is in debug mode?
                                ; status    : Object valid? (string)


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; Call GetProperty method from parent object
   ;; Note that execution continues even if retrieval of a baseobject
   ;; property fails.
   self->MGS_BaseVariable::GetProperty, _Extra=extra

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error retrieving variable properties!'
      RETURN
   ENDIF

   long_name = self.long_name
   units = self.units
   missing_value = *self.missing_value
   min_valid = *self.min_valid
   max_valid = *self.max_valid

   infoobject = self.infoObj

END

; -------------------------------------------------------------------------------------
; SetProperty:
; This method sets specific object values.

PRO MGS_Variable::SetProperty,  $
                   long_name=long_name,  $ ; A more complete description of the variable
                   units=units,          $ ; The physical unit of the variable
                   missing_value=missing_value, $
                   min_valid=min_valid,  $
                   max_valid=max_valid,  $
                   result=result,        $  ; result of Validate
                   _Extra=extra
                                ; Inherited keywords:
                                ; data      : The data values
                                ; dim1..dim8 : The dimension "values"
                                ; dims      : The sizes of the dimensions
                                ; tname     : The data type (string)
                                ; name      : The variable name
                                ; tname     : The data type (string)
                                ; uvalue    : a user-defined value
                                ; no_copy   : don't copy data when
                                ;             creating pointers
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; debug     : put object in debug mode

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; Call SetProperty method of BaseObject
   self->MGS_BaseVariable::SetProperty, result=result, _Extra=extra
   IF result NE 'OK' THEN BEGIN
      must_report = (result EQ 'NO_VALID_DATA') OR (result EQ 'UNKNOWN_STATUS')
      IF must_report OR self.debug GT 1 THEN BEGIN
         self->ErrorMessage, ['Error setting properties of parent object!', $
                              'Reason: '+result]
      ENDIF 
      IF result EQ 'NO_VALID_DATA' THEN RETURN
   ENDIF

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error setting variable properties!'
      RESULT = 'CANCELLED'
      RETURN
   ENDIF

   tname = self.tname
   IF tname EQ 'NOTHING' THEN tname = 'DOUBLE'

   ;; Test arguments
   IF N_Elements(long_name) GT 0 THEN self.long_name = long_name
   IF N_Elements(units) GT 0 THEN self.units = units

   IF N_Elements(missing_value) GT 0 THEN BEGIN
      umiss = missing_value[0]
      IF finite(umiss) THEN self->ConvertType, umiss, tname=tname
      *self.missing_value = umiss
   ENDIF

   IF N_Elements(min_valid) GT 0 THEN BEGIN
      umin = min_valid[0]
      self->ConvertType, umin, tname=tname
      *self.min_valid = umin
   ENDIF

   IF N_Elements(max_valid) GT 0 THEN BEGIN
      umax = max_valid[0]
      self->ConvertType, umax, tname=tname
      *self.max_valid = umax
   ENDIF

END


; -------------------------------------------------------------------------------------
; Cleanup:
; This method frees all data stored in the object.

PRO MGS_Variable::Cleanup

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   Ptr_Free, self.missing_value, self.min_valid, self.max_valid

   IF Obj_Valid(self.infoObj) THEN Obj_Destroy, self.infoObj

   self->MGS_Basevariable::Cleanup

END

; -------------------------------------------------------------------------------------
; Init:
; This method initializes the object values. See MGS_Basevariable for
; details. This object merely adds parameters for data description and
; sets the default data type to double.

FUNCTION MGS_Variable::Init,  $
                   data,                 $ ; The data values (type will be checked)
                   dim1,                 $ ; Values for first dimension (X)
                   dim2,                 $ ; Values for second dimension (Y)
                   dim3,                 $ ; Values for third dimension (Z or T)
                   dim4,                 $ ; ...
                   dim5,                 $ 
                   dim6,                 $ 
                   dim7,                 $ 
                   dim8,                 $ 
                   long_name=long_name,  $ ; A more complete description of the variable
                   units=units,          $ ; The physical unit of the variable
                   missing_value=missing_value,  $ ; The code for missing data
                   min_valid=min_valid,  $ ; The minimum valid data value
                   max_valid=max_valid,  $ ; The minimum valid data value
                   _Extra=extra        ; For future additions
                                ; Inherited keywords:
                                ; data      : The data values
                                ; dim1..dim8 : The dimension "values"
                                ; dims      : The sizes of the dimensions
                                ; tname     : The data type (string)
                                ; name      : The variable name
                                ; uvalue    : a user-defined value
                                ; no_dialog : Don't display message dialogs
                                ; debug     : Put object in debugging state



   ;; Initialize parent object first
   IF not self->MGS_BaseVariable::Init(data, $
                                       dim1, dim2, dim3, dim4, $
                                       dim5, dim6, dim7, dim8, $
                                       _Extra=extra) THEN RETURN, 0

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel  
      self->ErrorMessage, 'Error initializing variable object!'
      Ptr_Free, self.missing_value, self.min_valid, self.max_valid
      RETURN, 0
   ENDIF

   ;; Test arguments
   tname = self.tname
   IF tname EQ 'NOTHING' THEN tname = 'DOUBLE'
   IF N_Elements(long_name) EQ 0 THEN long_name = ''
   IF N_Elements(units) EQ 0 THEN units = ''
   IF N_Elements(missing_value) EQ 0 THEN BEGIN
      umiss = !Values.D_NaN
   ENDIF ELSE BEGIN
      umiss = missing_value[0]
      self->ConvertType, umiss, tname=tname
   ENDELSE

   IF N_Elements(min_valid) EQ 0 THEN BEGIN
      self->ConvertType, 0., min_valid=umin, tname=tname
   ENDIF ELSE BEGIN
      umin = min_valid[0]
      self->ConvertType, umin, tname=tname
   ENDELSE

   IF N_Elements(max_valid) EQ 0 THEN BEGIN
      self->ConvertType, 0., max_valid=umax, tname=tname
   ENDIF ELSE BEGIN
      umax = max_valid[0]
      self->ConvertType, umax, tname=tname
   ENDELSE

   ;; Initialize pointers
   self.missing_value = Ptr_New(/Allocate)
   self.min_valid = Ptr_New(/Allocate)
   self.max_valid = Ptr_New(/Allocate)

   ;; Populate object
   self.long_name = long_name
   self.units = units
   *self.missing_value = umiss
   *self.min_valid = umin
   *self.max_valid = umax

   RETURN, 1
END


; -------------------------------------------------------------------------------------
; This is the object definition. The MGS_Variable object inherits from
; MGS_Basevariable and MGS_Basevariable inherits from MGS_Baseobject.

PRO MGS_Variable__Define

   objectClass = { MGS_Variable,  $    ; The object class
                   long_name     : '',   $ ; A more complete description of the variable
                   units         : '',   $ ; The physical unit of the variable
                   missing_value : Ptr_New(), $ ; A value indicating missing data
                   min_valid     : Ptr_New(), $ ; The minimum valid data value
                   max_valid     : Ptr_New(), $ ; The maximum valid data value
                   infoObj       : Obj_New(), $ ; A GUI object for variable properties
                   INHERITS MGS_BaseVariable  $

                 }
                   

END



; -------------------------------------------------------------------------------------
; Example: 
;    This example demonstrates a few features of the Variable
; object, notably how to make a copy and how to retrieve data from it.

PRO Example, No_Plot=no_plot

;;  GOTO, test3

test1:
   ;; Create two dimension variables and one data variable
   lon  = obj_new('MGS_Variable', findgen(128), name='LON', $
                 long_name='Longitude', units='degrees_east', debug=1)
   lat  = obj_new('MGS_Variable', findgen(64), name='LAT', $
                 long_name='Latitude', units='degrees_north', debug=1)

   var1 = obj_new('MGS_Variable', dist(128,64), lon, lat,  $
                  name='Ozone', long_name='Ozone concentration', units='ppb', debug=1)

   ;; Create a second data variable as a clone of the first
   ;; Note: This variable will have independent dimension variables!
   var2 = var1->Copy()
;   var2 = obj_new('MGS_Variable', dist(128,64), lon, lat,  $
;                  name='Ozone', long_name='Ozone concentration', units='ppb', debug=2)
   var2->SetProperty, name='CO', long_name='Carbon monoxide concentration'

   ;; Put both objects into debugging mode (more verbose error messages)
   var1->setproperty, debug=1
   var2->setproperty, debug=1

   ;; Retrieve the name property from both variables to see that they are different:
   var1->getproperty,name=name1
   var2->getproperty,name=name2
   print,'Name 1 = ',name1,'  Name2 = ',name2

   ;; Invert the dimensions of the first variable
   lon->setproperty,data=reverse(findgen(128))
   lat->setproperty,data=reverse(findgen(64))

   ;; Print the first element of lon and lat for both variables:
   var1->getproperty,dim1=d1,dim2=d2
print,'******* retrieving lon1...'
   lon1 = d1->getData()
print,'******* retrieving lat1...'
   lat1 = d2->getData()
   var2->getproperty,dim1=d1,dim2=d2
print,'******* retrieving lon2...'
   lon2 = d1->getData()
print,'******* retrieving lat2...'
   lat2 = d2->getData()
   print,'First elements of :'
   print,'lon1 = ',lon1[0],'  lon2 = ',lon2[0]
   print,'lat1 = ',lat1[0],'  lat2 = ',lat2[0]

   ;; Scale the second data variable
   var2->Scale, factor=-1., offset=10.

   ;; Get all data and produce a contour plot
   IF Keyword_Set(no_plot) EQ 0 THEN BEGIN
      data = var2->GetData()
      c_level = findgen(41)*2.-20
      window, 0
      contour, data, lon2, lat2, level=c_level
   ENDIF 

   ;; Set variables missing value
   var2->SetProperty, missing_value=-999.9

   ;; Get data again, replace all negative values with -999.99
   data = var2->GetData(/positive, substitute=-999.99)
   IF Keyword_Set(no_plot) EQ 0 THEN BEGIN
      window, 1
      contour, data, lon2, lat2, level=c_level, min_val=0.
   ENDIF 

   ;; Set all -999.99 values to f_nan and store data back
   wneg = var2->Filter(/negative, result=result)
   IF result EQ 'OK' THEN BEGIN
      data[wneg] = !Values.F_NaN
      var2->SetProperty, data=data
   ENDIF

   ;; Now get all data unfiltered and produce a contour plot
   IF Keyword_Set(no_plot) EQ 0 THEN BEGIN
      data = var2->GetData()
      c_level = findgen(41)*2.-20
      window, 2
      contour, data, lon2, lat2, level=c_level
   ENDIF 

   ;; Show variable properties
   var1->ShowInfo, /block
   var2->ShowInfo, /block

   ;; Clean up
   Obj_Destroy, d1    ; copy of lon in var2
   Obj_Destroy, d2    ; copy of lat in var2
   Obj_Destroy, lon
   Obj_Destroy, lat

   Obj_Destroy, var1
   Obj_Destroy, var2


test2:

   ;; Second series: try out several filter criteria
   ;; 1. All data valid
   test  = obj_new('MGS_Variable', findgen(11)-5., name='TEST', debug=1)
   print,'Data = ',test->GetData(),format='(A22,11F8.1)'
   print,'/Finite_Only: ',test->Filter(/finite_only),format='(A22,11(I5,3X))'
   print,'/Positive: ',test->Filter(/positive),format='(A22,11(I5,3X))'
   print,'/Negative: ',test->Filter(/negative),format='(A22,11(I5,3X))'
   print,'/Non_Zero: ',test->Filter(/non_zero),format='(A22,11(I5,3X))'
   print,'/Negative, /Invert: ',test->Filter(/negative, /invert),format='(A22,11(I5,3X))'
   print,'/Non_Zero, /Invert: ',test->Filter(/non_zero, /invert),format='(A22,11(I5,3X))'
   print,'/Non_Missing: ',test->Filter(/non_missing),format='(A22,11(I5,3X))'
   print,'/Valid_Only: ',test->Filter(/valid_only),format='(A22,11(I5,3X))'
   print,'/Positive, /Valid_Only: ',test->Filter(/positive, /valid_only),format='(A22,11(I5,3X))'
   print

   ;; 2. Some data set to missing
   data = test->GetData()
   data[[2,4,6,8,11,13,15]] = !Values.F_NaN
   test->SetProperty, data=data   
   print,'Data = ',test->GetData(),format='(A22,11F8.1)'
   print,'/Finite_Only: ',test->Filter(/finite_only),format='(A22,11(I5,3X))'
   print,'/Positive: ',test->Filter(/positive),format='(A22,11(I5,3X))'
   print,'/Negative: ',test->Filter(/negative),format='(A22,11(I5,3X))'
   print,'/Non_Zero: ',test->Filter(/non_zero),format='(A22,11(I5,3X))'
   print,'/Negative, /Invert: ',test->Filter(/negative, /invert),format='(A22,11(I5,3X))'
   print,'/Non_Zero, /Invert: ',test->Filter(/non_zero, /invert),format='(A22,11(I5,3X))'
   print,'/Non_Missing: ',test->Filter(/non_missing),format='(A22,11(I5,3X))'
   print,'/Valid_Only: ',test->Filter(/valid_only),format='(A22,11(I5,3X))'
   print,'/Positive, /Valid_Only: ',test->Filter(/positive, /valid_only),format='(A22,11(I5,3X))'
   print

   ;; 3. Min_Valid and max_valid keyword set
   test->SetProperty, min_valid=-4., max_valid=6.
   print,'* Min_Valid = -4,  Max_Valid = 6'
   print,'Data = ',test->GetData(),format='(A22,11F8.1)'
   print,'/Finite_Only: ',test->Filter(/finite_only),format='(A22,11(I5,3X))'
   print,'/Positive: ',test->Filter(/positive),format='(A22,11(I5,3X))'
   print,'/Negative: ',test->Filter(/negative),format='(A22,11(I5,3X))'
   print,'/Non_Zero: ',test->Filter(/non_zero),format='(A22,11(I5,3X))'
   print,'/Negative, /Invert: ',test->Filter(/negative, /invert),format='(A22,11(I5,3X))'
   print,'/Non_Zero, /Invert: ',test->Filter(/non_zero, /invert),format='(A22,11(I5,3X))'
   print,'/Non_Missing: ',test->Filter(/non_missing),format='(A22,11(I5,3X))'
   print,'/Valid_Only: ',test->Filter(/valid_only),format='(A22,11(I5,3X))'
   print,'/Positive, /Valid_Only: ',test->Filter(/positive, /valid_only),format='(A22,11(I5,3X))'
   print

   ;; 4. Missing_Value keyword set to -999.99, and two values changed accordingly
   test->SetProperty, missing_value=-999.99
   print,'* Missing_Value = -999.99  (Min_Valid = -4,  Max_Valid = 6)'
   data[[8,9]] = -999.99
   test->SetProperty, data=data
   print,'Data = ',test->GetData(),format='(A22,11F8.1)'
   print,'/Finite_Only: ',test->Filter(/finite_only),format='(A22,11(I5,3X))'
   print,'/Positive: ',test->Filter(/positive),format='(A22,11(I5,3X))'
   print,'/Negative: ',test->Filter(/negative),format='(A22,11(I5,3X))'
   print,'/Non_Zero: ',test->Filter(/non_zero),format='(A22,11(I5,3X))'
   print,'/Negative, /Invert: ',test->Filter(/negative, /invert),format='(A22,11(I5,3X))'
   print,'/Non_Zero, /Invert: ',test->Filter(/non_zero, /invert),format='(A22,11(I5,3X))'
   print,'/Non_Missing: ',test->Filter(/non_missing),format='(A22,11(I5,3X))'
   print,'/Valid_Only: ',test->Filter(/valid_only),format='(A22,11(I5,3X))'
   print,'/Positive, /Valid_Only: ',test->Filter(/positive, /valid_only),format='(A22,11(I5,3X))'
   print,'/Negative, /Valid_Only: ',test->Filter(/negative, /valid_only),format='(A22,11(I5,3X))'
   print,'/Negative, /Non_Missing: ',test->Filter(/negative, /non_missing),format='(A22,11(I5,3X))'
   print

   Obj_Destroy, test


test3:
   ;; Third series: try out arithmetics
   ;; 1. All data valid
   test  = obj_new('MGS_Variable', findgen(11)-5., name='TEST', debug=1)
   print,'Mean of -5 .. +5 : ',test->Mean(),' {reference = ',Mean(findgen(11)-5.),'}'
   print,'Mean of -5 .. -1 : ',test->Mean(test->GetData(/negative)), $
      ' {reference = ',Mean([-5.,-4.,-3.,-2.,-1.]),'}'
   print,'Mean of -5 ..  0 : ',test->Mean(test->GetData(/positive,/invert)), $
      ' {reference = ',Mean([-5.,-4.,-3.,-2.,-1.,0.]),'}'
   ;; 2. Missing_Value keyword set to -999.99, and two values changed accordingly
   test->SetProperty, missing_value=-999.99
   print,'* Missing_Value = -999.99'
   data = test->GetData()
   data[[8,9]] = -999.99
   test->SetProperty, data=data
   print,'Data = ',test->GetData(),format='(A22,11F8.1)'
   print,'Mean of data: ',test->Mean(), $
      ' {reference = ',Mean([-5.,-4.,-3.,-2.,-1.,0.,1.,2.,5.]),'}'

   ;; 3. Min_Valid keyword set to 0.
   test->SetProperty, min_valid = 0.
   print,'* Min_Valid = 0.  (Missing_Value = -999.99)'
   print,'Data = ',test->GetData(),format='(A22,11F8.1)'
   print,'Mean of data: ',test->Mean(),  $
      ' {reference = ',Mean([0.,1.,2.,5.]),'}'
   print,'Mean of data (with subs=0.): ',test->Mean(min_subs=0.),  $
      ' {reference = ',Mean([0.,0.,0.,0.,0.,0.,1.,2.,5.]),'}'

   Obj_Destroy, test

   ;; 4. Try a 2-D array and extract mean along 1st dimension
   test = obj_new('MGS_Variable', findgen(5,5), name='TEST2D', debug=1)
   print,'Mean values:'
   print,test->Mean()
   print,test->Mean(dimension=1,nvalid=n),n
   print,test->Mean(dimension=2,nvalid=n),n

   print,'Median, Min, Max:'
   print,test->Median()
   print,test->Min()
   print,test->Max()

   test->SetProperty, min_valid = 0.
   print,'* Min_Valid = 0.'
;;   print,test->Mean(dimension=1,nvalid=n),n   ; should give error unless you have positive data only

   test->SetProperty, missing_value=-999.99
   print,'* Missing_Value = -999.99 (and min_valid = 0.)'
   data = test->GetData()
   data[[8,9]] = -999.99
   test->SetProperty, data=data
   print,test->Mean()
;;   print,test->Mean(dimension=1,nvalid=n),n   ; should give error!

   ;; Now compute the sine of data in degrees
   data = findgen(37.)*10.
   test->SetProperty,data=data
   print,'Data = ',test->GetData()
   ;; convert to radian
   test->Scale, factor=!DPI/180.
   print,'Data = ',test->GetData()
   res = test->CallFunction(result=result,name='Sin')
   print,'Sin(data)=',res

   ;; compute percentiles of degree values
   test->Scale, factor=180./!DPI
   res = test->CallFunction(result=result,name='Percentiles')
   print,'Percentiles(data[degrees])=',res
   res = test->CallFunction(result=result,name='Percentiles',value=[0.33333333,.66666666])
   print,'Percentiles(data[degrees])=',res

   ;; call an undefined function
;;   res = test->CallFunction(result=result,name='JDIjwucoKDHJ')
;;   print,'JDIjwucoKDHJ(data)=',res

   Obj_Destroy, test

   
END
