;+
;
; Notes:
;   (1) Variables and dimensions are linked through a list of
;   dimension variable names. Usually, the dimension variables are
;   so-called coordinate variables (which must be monotonically
;   increasing or decreasing) and have the same name as the
;   respective dimension. However, you can also choose any other
;   1-dimensional variable as dimension variable. This simplifies a
;   change of coordinates e.g. in plotting. For example, if you have
;   two 1-dimensional variables with the dimension time, 'time' and
;   'date', you may select either 'time' or 'date' as dimension
;   variable (default is 'time').
;
;-



FUNCTION MGS_Varlist::GetDefaultDimVarnames

; returns string array with the variable names that have closest match
; to dimension names:
; (1) default is to use variable with identical name
; (2) next find variable that contains dimname or which is contained
; in dimname and has the correct dimension
; (3) next find first variable that has the correct dimension
; (4) if all fails return empty string

END


PRO MGS_Varlist::SetDefaultDimVarnames

; Resets the dimvarnames to theirdefault values as returned by
; GetDefaultDimVarnames. If a dimvarname is empty, a new variable with
; the name of the dimension is created with monotonically increasing
; values starting with 0.

END


FUNCTION MGS_Varlist::GetDimVarnames, dimname

; Returns a string array with the names of all variables that could be
; used as dimension variable. The first entry is the default dimension
; variable name.

END


PRO MGS_Varlist::SetDimVarnames, names, $
               position=position, $  ; The position index to be changed
               dimnames=dimnames     ; The dimension names to be changed

; This method overwrites the dimension variable name for one or
; several dimensions. If neither position nor dimnames is given, the
; names are filled beginning with index 0. The routine reports an
; error if there is no variable for an element of names or if that
; variable cannot serve as dimension variable for the respective
; dimension. 

END



PRO MGS_Varlist::AddVariable, varobject, $
               success=success    ; A flag indicating the successful operation

; This method adds a new variable to the variable container. 
; Tests:
; - duplicate name
; - matching dimensions

END


PRO MGS_Varlist::Append, varlist

; This method appends the variables of another varlist to the present
; list
; Details:
; - how to recognize dimension for extension
; - check consistency of dimensions and variables
; - check consistency of extended dimension (monotony)
; - if variable is missing in one object: either delete or set to
;   missing_value


END


FUNCTION MGS_Varlist::Extract, variablenames=variablenames, $
                    _Extra=extra        ; use extra keyword to extract limited dim range


END


; -----------------------------------------------------------------------------
; Init:
;   This method initializes the variable list object. The variable
; objects may either be passed in the initialization phase or added
; later with the AddVariables method. The required dimensions must be
; defined before any variable can be stored.

FUNCTION MGS_Varlist::INIT, $
                    Title=title,  $ ; A title string for widgets
                    $           ; describing the contents of the
                    $           ; file. Normally, this information is
                    $           ; extracted from the title tag of the
                    $           ; gattr structure.
                    Dimnames=dimnames,  $ ; A string array with the 
                    $           ; dimension names
                    Dimsizes=dimsizes, $ ; A int or lon array with the
                    $           ; dimension sizes (must have same number of 
                    $           ; elements as dimnames)
                    Vars=vars, $ ; An object array with variables 
                    _Ref_Extra=extra ; For inherited and future keywords
                                ;
                                ; Inherited keywords:
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value                    

   ;; Initialize parent object first
   IF not self->MGS_BaseObject::Init(_Extra=extra) THEN RETURN, 0
   
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error initializing object!'
      RETURN, 0
   ENDIF
   
   ;; Set title
   IF N_Elements(title) GT 0 THEN $
     self.title = StrTrim(title, 2)  $
   ELSE $
     self.title = ' '
  
   ;; Initialize the dimensions
   self.dimnames = ''
   self.dimsizes = 0L
   IF N_Elements(dimnames) NE N_Elements(dimsizes) THEN BEGIN
      self->ErrorMessage, 'Dimnames must have same number of elements as dimsizes!'
      RETURN, 0
   ENDIF 
   maxd = ( Size(self.dimnames,/Dimensions) )[0]
   IF N_Elements(dimnames) GT maxd THEN BEGIN
      self->ErrorMessage, [ 'Too many dimensions! ', $
                            'Increase MAX_DIMENSIONS in MGS_Varlist__Define.', $
                            'Currently '+ StrTrim(maxd,2) ]
      RETURN, 0
   ENDIF 
   IF N_Elements(dimnames) GT 0 THEN BEGIN
      self.dimnames = dimnames
      self.dimsizes = long(dimsizes)
   ENDIF 

   ;; Treat dimension names case-insensitve: Convert all to uppercase
   self.dinames = StrUpcase(self.dimnames)

   ;; Initialize container to store variable objects
   self.vars = Obj_New('MGS_Container') 

   ;; Store variables in container if any are passed
   IF Obj_Valid(vars[0]) THEN BEGIN
      FOR i=0L, N_Elements(vars)-1 DO BEGIN
         self->AddVariable, vars[i], success=success
         IF success EQ 0 THEN RETURN, 0    
                                ; Error message in AddVariable
                                ; Hopefully, IDL takes care of cleanup
      ENDFOR 

      ;; Identify the dimension variables
      self->SetDefaultDimVarNames
   ENDIF 

   RETURN, 1
END

 
; -----------------------------------------------------------------------------
; MGS_Varlist__Define:
; This is the object definition for the variable list object. It
; inherits from MGS_BaseObject the abilities to set and query an
; object name and a uvalue. The base object also provides a general
; method for display of error messages which can be directed to a
; message dialog or to the log screen via the no_dialog flag, and
; a generic copy method to create a clone.

PRO MGS_Varlist__Define

   MAX_DIMENSIONS = 32    ; maximum number of independent dimensions

   objectClass = { MGS_Varlist, $ ; The object class
           ;;; data information
           title: '',  $      ; A title describing the data set
           dimnames: StrArr(MAX_DIMENSIONS), $ ; The names of the dimensions
           dimsizes: LonArr(MAX_DIMENSIONS), $ ; The size of the dimensions
           dimvarnames: StrArr(MAX_DIMENSIONS),$ ; The (current) dimension variable names
           vars: Obj_New(), $ ; Container with variable objects
           ;;; heritage
           INHERITS MGS_BaseObject  $ ; provides basic general properties
     }

END
