function geov_extractor::lat_weighted_ave,  counts, offsets

  ; computes a global average from the zonal average

  ndims = n_elements(counts)

  vardims = self->variable_dimensions(self.vid)

  ilat = vardims.latdim
  ilon = vardims.londim
  ilev = vardims.levdim
  itim = vardims.timdim

  nlat = counts[ilat] 
  if (ilon ge 0) then nlon = counts[ilon] else nlon = 1
  if (ilev ge 0) then nlev = counts[ilev] else nlev = 1
  if (itim ge 0) then ntim = counts[itim] else ntim = 1

  d2coord = self.ncfiles->get_d2coord()  ; pointer dimension id --> coordinate number
  did = self.ncfiles->get_dimid_of_var(ilat, self.vid) ; coordinate dimension id
  ic = (*d2coord)[did]                          ; coordinate number
  lats = (*(self.ncfiles->get_coord(ic)))*!pi/180.

  latsi = [-!pi*.5,((lats + shift(lats,1))*.5)[1:nlat-1],!pi*.5]
  dlats = (latsi - shift(latsi,1))[1:nlat]

  data = *self.data
  refarr = counts
  if (ilon ge 0) then refarr[ilon] = 1
  data = reform( data, refarr )

  sum = fltarr(nlev,ntim)

  c = intarr(ndims,2)
  if (ilon ge 0) then c[ilon,*]=0
  c[ilat,0] = 0
  c[ilat,1] = nlat-1

  for t=0,ntim-1 do begin
    for k=0,nlev-1 do begin

      if (itim ge 0) then c[itim,*] = t
      if (ilev ge 0) then c[ilev,*] = k
 
      indices = '[' 
      for ic=0,ndims-2 do indices=indices+string(c[ic,0])+':'+string(c[ic,1])+','
      indices=indices+string(c[ic,0])+':'+string(c[ic,1])+']'

      command = 'sum[k,t] = total(data'+ indices $
                  +' * cos(lats) * dlats )'
      command = strcompress(command,/remove_all)
      success = EXECUTE(command)


      if (not success) then begin
        txt = ['Error occurred in geov_extractor::lat_weighted_ave... EXECUTE(command)',$
                'command : '+command ]
        xxx = dialog_message( txt, /error )
        print,txt[0]
        print,txt[1]
        return,0
      endif

    endfor
  endfor

  sum = sum / TOTAL( cos(lats)*dlats )

  return, sum

end

function geov_extractor::extract_at_const_press, offsets, counts, press
  @geov_pars
  f = *self.data

  if ( self.ncfiles->var_depends_on_ilevi(self.vid) ) then $
    press_ptr = self->pressure_field('hyai','hybi', offsets, counts) $
  else $
    press_ptr = self->pressure_field('hyam','hybm', offsets, counts)

  p = *press_ptr

  vardims = self->variable_dimensions(self.vid)

  ilat = vardims.latdim
  ilon = vardims.londim
  ilev = vardims.levdim

  dsizes = *(self.ncfiles->get_dsizes())

  nlats = dsizes[ self.ncfiles->get_dimid_of_var(ilat,self.vid) ]
  nlons = dsizes[ self.ncfiles->get_dimid_of_var(ilon,self.vid) ]
  nlevs = dsizes[ self.ncfiles->get_dimid_of_var(ilev,self.vid) ]

  x = make_array( 3, /long )
  c1 = make_array( 3, /long )
  c2 = make_array( 3, /long )
  c3 = make_array( 3, /long )

  x[ilat] = nlats
  x[ilon] = nlons
  x[ilev] = 1

  g = make_array(dimension=x, /float)

  for i=0, nlats-1 do begin
    for j=0, nlons-1 do begin

      k = 0L
      c2[ilat] = i
      c2[ilon] = j
      c2[ilev] = k

      while ( (press gt p[c2[0],c2[1],c2[2]]) and (k lt nlevs) ) do begin

        k = k+1
        if (k eq nlevs) then break
        c2[ilev] = k

      endwhile
      
      c1[ilat] = i
      c1[ilon] = j
      c1[ilev] = k-1
      c3[ilat] = i
      c3[ilon] = j
      c3[ilev] = 0

      if ( (k eq 0) or (k ge nlevs)) then begin
        g[c3[0],c3[1],c3[2]] = NULL_VALUE
      endif else begin

        fact1 = (p[c1[0],c1[1],c1[2]] - press)/(p[c1[0],c1[1],c1[2]] - p[c2[0],c2[1],c2[2]])
        fact2 = (press - p[c2[0],c2[1],c2[2]])/(p[c1[0],c1[1],c1[2]] - p[c2[0],c2[1],c2[2]])

        g[c3[0],c3[1],c3[2]] = fact2*f[c1[0],c1[1],c1[2]] + fact1*f[c2[0],c2[1],c2[2]]
      endelse

    endfor
  endfor

  result = reform( g )
  return, result

end

; method to extract data from netCDF file according to user-supplied actions
; KEYWORDS: scale_factor=1 --> determine best scale factor for these data
;           scale_factor!=1 --> use input scale factor instead
; mgs 22 April 2002: added functionality to process U and V wind data
; (NO AVERAGING YET)

;PRO geov_extractor::extract_data, hourglass=hourglass
function geov_extractor::extract_data, hourglass=hourglass, ex_type=ex_type, simple=simple
    @geov_pars
    if (n_elements(hourglass) gt 0) then widget_control, hourglass=hourglass

    ; extract information from netCDF file
    ncoords = self.ncfiles->get_ncoords()
    vndim = (*(self.ncfiles->get_vndims()))[self.vid]
    dsizes = self.ncfiles->get_dsizes()
    vnames = self.ncfiles->get_vnames()
    d2coord = self.ncfiles->get_d2coord()  ; pointer dimension id --> coordinate number
    cvars = self.ncfiles->get_cvars() ; coordinate number --> variable id

    dobson = 0

    if ( strcmp(string((*vnames)[self.vid]), 'O3', 2, /fold_case) or $
         strcmp(string((*vnames)[self.vid]), 'OX', 2, /fold_case) ) then dobson=1

    ; allocate memory
    counts = ptr_new(lonarr(vndim))
    offsets = ptr_new(lonarr(vndim))

    ; loop over variable dimensions, get reference to xs, ys,
    ; prepare to extract data
    iX=-1
    iY=-1
    vactions = ptr_new(lonarr(vndim)) ; temporary storage for variable actions ordered as variable dimensions

    for idim=0, vndim-1 do begin    ; dimension index

       did = self.ncfiles->get_dimid_of_var(idim, self.vid) ; coordinate dimension id
       ic = (*d2coord)[did]                          ; coordinate number
       cvid= (*cvars)[ic]                            ; coordinate variable id
       action = (*self.actions)[ic]
       (*vactions)[idim] = action


       CASE action OF

            XAXIS: BEGIN
                     self.xs = self.ncfiles->get_coord(ic)
                     (*offsets)[idim] = 0
                     (*counts)[idim] = n_elements(*self.xs)
                     iX=idim
                   END

            YAXIS: BEGIN
                     self.ys = self.ncfiles->get_coord(ic)
                     (*offsets)[idim] = 0
                     (*counts)[idim] = n_elements(*self.ys)
                     iY=idim
                   END

             AVERAGE: BEGIN
                       (*offsets)[idim] = 0
                       (*counts)[idim]  = (*dsizes)[did]
                      END

             WEIGHTEDAVE: BEGIN
                       (*offsets)[idim] = 0
                       (*counts)[idim]  = (*dsizes)[did]
                      END

             CONSTPRESS: BEGIN
                       (*offsets)[idim] = 0
                       (*counts)[idim]  = (*dsizes)[did]
                       ;(*counts)[idim]  = 1L
                      END

             ELSE: BEGIN
                     (*offsets)[idim] = action ; start from this index
                     (*counts)[idim]  = 1L
                   END
       ENDCASE

    endfor ; loop over variable dimensions


    ; free memory from previous call
    if (ptr_valid(self.data)) then ptr_free, self.data
    if (ptr_valid(self.uwind)) then ptr_free, self.uwind
    if (ptr_valid(self.vwind)) then ptr_free, self.vwind

    ; extract data from file
    self.data = self.ncfiles->get_data(self.vid, *offsets, *counts)

    ; compile the geov_extractor__extract.pro file
    self->extract

    self.gui->set_totcol_flag,0L
    self.gui->set_numden_flag,0L

;    if ( self.colflag ne 0 ) then  begin
;      ok = self->check_tot_col()
;      if (ok) then begin
;        if ( self.colflag eq EX_TOTAL_COL ) then begin
;          self.data = self->int_col_dens( *offsets, *counts )
;        endif else begin
;          llv = self.gui->get_lev()
;          self.data = self->int_col_dens( *offsets, *counts , above_lev=llv)
;        endelse
;        self.gui->set_totcol_flag,self.colflag
;      endif else begin
;        return, 0
;      endelse
;    endif else if ( self.numflag ne 0 ) then begin
;      self.data = self->number_density( *offsets, *counts )
;      self.gui->set_numden_flag,1L
;    endif else if ( self.tmassflag ne 0 ) then begin
;      self.data = self->tracer_mass()
;      self.gui->set_tmass_flag,1L
;    endif

    if n_elements(ex_type) gt 0 then begin
      ;if (ptr_valid(self.data)) then ptr_free, self.data
      if ex_type ne EX_SIMPLE then begin
        CASE ex_type OF
          EX_NUM_DENS: $
            begin
              self.data = self->number_density( *offsets, *counts )
              self.gui->set_numden_flag,1L
            end
          EX_TOTAL_COL: $
            begin
              ok = self->check_tot_col()
              if (ok) then begin
                self.data = self->int_col_dens( *offsets, *counts, dobson=dobson )
                self.gui->set_totcol_flag,self.colflag
              endif else begin
                return, 0
              endelse
            end
          EX_COL_ABOVE: $
            begin
              ok = self->check_tot_col()
              if (ok) then begin
                llv = self.gui->get_lev()
                self.data = self->int_col_dens( *offsets, *counts , above_lev=llv, dobson=dobson )
                self.gui->set_totcol_flag,self.colflag
              endif else begin
                return, 0
              endelse
            end
          EX_TROP_COL: $
            begin
              ok = self->check_tot_col()
              if (ok) then begin
                llv = self.gui->get_lev()
                self.data = self->trop_col_dens( *offsets, *counts, dobson=dobson  )
                self.gui->set_totcol_flag,self.colflag
              endif else begin
                return, 0
              endelse
            end
          EX_ABOVETROP_COL: $
            begin
              ok = self->check_tot_col()
              if (ok) then begin
                llv = self.gui->get_lev()
                self.data = self->above_trop_col_dens( *offsets, *counts, dobson=dobson  )
                self.gui->set_totcol_flag,self.colflag
              endif else begin
                return, 0
              endelse
            end
          EX_SUMMATION: $
            begin
              gui = self.gui
              selected_flds = gui->get_summed_flds()
              sum_coefs = gui->get_sum_coefs()
              show = gui->get_show_sum_gui()
              self.data = operator_summation( gui, group_leader=gui->get_base(),name=name ,selected_flds=selected_flds, coefs=sum_coefs, show=show )
              gui->set_summed_flds, selected_flds
              gui->set_sum_coefs, sum_coefs
            end
          EX_TOT_MASS: $
            begin
              self.data = self->tracer_mass()
              self.gui->set_tmass_flag,1L
            end
          ELSE: 
        ENDCASE
      endif
    endif

    if ( not ptr_valid(self.data) ) then return, 0

  if ( not keyword_set( simple ) ) then begin
    ; extract U and V wind component if desired and present
    uvid = self.ncfiles->get_iuvar()
    vvid = self.ncfiles->get_ivvar()
    IF uvid GE 0 THEN BEGIN
       winddim = (*(self.ncfiles->get_vndims()))[uvid]
    ENDIF ELSE BEGIN
       winddim = -1L
    ENDELSE
    if not ( self.ncfiles->var_depends_on_ilevi(self.vid) ) then begin
      IF uvid GE 0 AND vvid GE 0 AND winddim EQ vndim THEN BEGIN
        self.uwind = self.ncfiles->get_data(uvid, *offsets, *counts)
        self.vwind = self.ncfiles->get_data(vvid, *offsets, *counts)
      ENDIF
    endif else begin
      self.uwind = ptr_new()
      self.vwind = ptr_new()
    endelse

    ; substitute missing data --> NULL_VALUE
    vnull = self.ncfiles->get_vnull(self.vid)
    if (finite(vnull)) then begin
        indexes = where(FLOAT(*self.data) eq vnull, n)
        if (n gt 0) then begin
            (*self.data)[indexes] = NULL_VALUE
        endif
    endif

    ; average over requested dimensions (NOTE: may be more than 1)
    iav = 1 ; index of dimension on which to perform average, starting at 1
    for idim=0, vndim-1 do begin
        if ((*vactions)[idim] eq AVERAGE) then begin
          ; CGB - Calculating averages with data that includes NaN.
          tmp = total(*self.data,iav,/NaN) ; check for missing values
          npts = total(finite(*self.data),iav)
          bad = where(npts eq 0., nbad)
          tmp = tmp / (npts>1)
          if (nbad gt 0) then tmp[bad] = NULL_VALUE
;          tmp = total(*self.data,iav) ; check for missing values
;          did = self.ncfiles->get_dimid_of_var(idim, self.vid) ; coordinate dimension id
;          tmp = tmp/float( (*dsizes)[did] )
          ptr_free, self.data ; free memory
          self.data = ptr_new(tmp,/NO_COPY)
        endif else begin
          iav = iav + 1 ; do not average over this dimension
        endelse
    endfor

    for idim=0, vndim-1 do begin
        if ((*vactions)[idim] eq WEIGHTEDAVE) then begin
          tmp =  self->lat_weighted_ave( *counts, *offsets )
          ptr_free, self.data ; free memory
          self.data = ptr_new(tmp,/NO_COPY)
          break
        endif
    endfor

    for idim=0, vndim-1 do begin
      if ((*vactions)[idim] eq CONSTPRESS) then begin
       oldPressPtr =  self.gui->get_const_press()
        if ( ptr_valid( oldPressPtr ) ) then begin
          pressurePtr = pressure_entry(group_leader=self.gui->get_base(),pressure=*oldPressPtr) 
        endif else begin
          pressurePtr = pressure_entry(group_leader=self.gui->get_base())
        endelse

        if (ptr_valid(pressurePtr)) then begin
          tmp = self->extract_at_const_press( *offsets, *counts, *pressurePtr*100.)
          self.data = ptr_new(tmp,/NO_COPY)
          self.gui->set_const_press, pressurePtr 
        endif else return, 0
      endif
    endfor

    ; substitute back NaN-->vnull
    if (finite(vnull)) then begin
        indexes = finite(*self.data)  ; array of 1, 0
        non_finite = where(indexes eq 0, n)
        if (n gt 0) then (*self.data)[non_finite] = vnull
    endif

    ; get rid of redundant dimensions
    xyz = *self.data
    tmp = reform(*self.data)
    ptr_free, self.data ; free memory
    self.data = ptr_new(tmp,/NO_COPY)


    IF ptr_valid(self.uwind) AND ptr_valid(self.vwind) THEN BEGIN
       tmp = reform(*self.uwind)
       ptr_free, self.uwind     ; free memory
       self.uwind = ptr_new(tmp,/NO_COPY)

       tmp = reform(*self.vwind)
       ptr_free, self.vwind     ; free memory
       self.vwind = ptr_new(tmp,/NO_COPY)
    ENDIF

    ; optionally, swap the two indexes
    if (iY ge 0 and iX ge 0 and iX gt iY) then begin
        tmp = transpose(*self.data)
        ptr_free, self.data ; free memory
        self.data = ptr_new(tmp,/NO_COPY)

        IF ptr_valid(self.uwind) AND ptr_valid(self.vwind) THEN BEGIN
           tmp = transpose(*self.uwind)
           ptr_free, self.uwind ; free memory
           self.uwind = ptr_new(tmp,/NO_COPY)

           tmp = transpose(*self.vwind)
           ptr_free, self.vwind ; free memory
           self.vwind = ptr_new(tmp,/NO_COPY)
        ENDIF
    endif

  endif

    ; deallocate memory
    ptr_free, offsets
    ptr_free, counts
    ptr_free, vactions

    return, 1

END
