;======================================================================
;
; Class Geov_Multifiles
;
; This class object manages reading data from multiple netCDF files
; with one control panel.
;
;======================================================================


;======================================================================
; Method Geov_Multifiles::is_cflag_set
;
;======================================================================
Function geov_multifiles::is_cflag_set, ilev,flag
  return, (*self.files)[0]->is_cflag_set(ilev,flag)
End


;======================================================================
; Method Geov_Multifiles::get_coord
;
; Returns all values of the specified coordinate in the current set of
; netCDF files.
;
; Parameters:
;   ic: Coordinate index
;======================================================================
Function geov_multifiles::get_coord, ic
  ; If ic is the time coord we need to return all times.
  ; Otherwise we use the get_coord method of the first file object
  file = (*self.files)[0]
  itim = file->get_itim()

  if ( ic eq itim ) then begin
;    x = self.alltimes
    x = self->get_times()
  endif else  begin
    x = file->get_coord(ic)
  endelse
  return, x
End

;======================================================================
; Method Geov_Multifiles::get_fname
;
; Returns the name of the first file in the set
;======================================================================
Function geov_multifiles::get_fname
  return, (*self.files)[0]->get_fname()
End

;======================================================================
; Method Geov_Multifiles::get_fid
;
; Returns the id of the first file in the set
;======================================================================
Function geov_multifiles::get_fid
  return, (*self.files)[0]->get_fid()
End

;======================================================================
; Method Geov_Multifiles::get_dimid_of_var
;
;======================================================================
Function geov_multifiles::get_dimid_of_var, idim, vid
  return, (*self.files)[0]->get_dimid_of_var(idim, vid)
End

;======================================================================
;
;======================================================================
Function geov_multifiles::var_depends_on_ilevi, vid
  return, (*self.files)[0]->var_depends_on_ilevi(vid)
End

;======================================================================
; Method Geov_Multifiles::get_ncoords
;
;======================================================================
Function geov_multifiles::get_ncoords
  return, (*self.files)[0]->get_ncoords()
End

;======================================================================
; Method Geov_Multifiles::get_d2coord
;
;======================================================================
Function geov_multifiles::get_d2coord
  return, (*self.files)[0]->get_d2coord()
End

;======================================================================
; Method Geov_Multifiles::get_p2coords
;
;======================================================================
Function geov_multifiles::get_p2coords
  return, (*self.files)[0]->get_p2coords()
End

;======================================================================
; Method Geov_Multifiles::get_iuvar
;
;======================================================================
Function geov_multifiles::get_iuvar
  return, (*self.files)[0]->get_iuvar()
End

;======================================================================
; Method Geov_Multifiles::get_ivvar
;
;======================================================================
Function geov_multifiles::get_ivvar
  return, (*self.files)[0]->get_ivvar()
End

;======================================================================
; Method Geov_Multifiles::get_cdims
;
;======================================================================
Function geov_multifiles::get_cdims
  return, (*self.files)[0]->get_cdims()
End

;======================================================================
; Method Geov_Multifiles::get_cvars
;
;======================================================================
Function geov_multifiles::get_cvars
  return, (*self.files)[0]->get_cvars()
End

;======================================================================
; Method Geov_Multifiles::get_vndims
;
;======================================================================
Function geov_multifiles::get_vndims
  return, (*self.files)[0]->get_vndims()
End

;======================================================================
; Method Geov_Multifiles::get_ndims
;
;======================================================================
Function geov_multifiles::get_ndims
  return, (*self.files)[0]->get_ndims()
End

;======================================================================
; Method Geov_Multifiles::get_dnames
;
;======================================================================
Function geov_multifiles::get_dnames
  return, (*self.files)[0]->get_dnames()
End

;======================================================================
; Method Geov_Multifiles::get_vnames
;
;======================================================================
Function geov_multifiles::get_vnames
  return, (*self.files)[0]->get_vnames()
End

;======================================================================
; Method Geov_Multifiles::get_vnull
;
;======================================================================
Function geov_multifiles::get_vnull, vid
  return, (*self.files)[0]->get_vnull( vid)
End

;======================================================================
; Method Geov_Multifiles::get_vunits
;
;======================================================================
Function geov_multifiles::get_vunits
  return, (*self.files)[0]->get_vunits()
End

;======================================================================
; Method Geov_Multifiles::get_dsizes
;
; Returns reference to an array which contains the dimension sizes,
; i.e., the number of level, latitude, longitude, and time grid points
; contained in the set of netCDF files.
;======================================================================
Function geov_multifiles::get_dsizes

  ; we need to get the time index of the dsizes array
  file = (*self.files)[0]
  dsizes = file->get_dsizes()
  cdims = *(file->get_cdims())

  itim = file->get_itim()
  
  if ( itim ge 0 and itim lt n_elements(*dsizes) ) then begin
   tid = cdims[itim]

   ; the time value of the dsizes array needs to be adjusted to account
   ; for all the time steps in the current set of files
   tot = total(*self.ntimes)
   (*dsizes)[tid] = tot
  endif

  return,  dsizes
End

;======================================================================
; Method Geov_Multifiles::get_lons
;
;======================================================================
Function geov_multifiles::get_lons
  return, (*self.files)[0]->get_lons()
End

;======================================================================
; Method Geov_Multifiles::get_lats
;
;======================================================================
Function geov_multifiles::get_lats
  return, (*self.files)[0]->get_lats()
End

;======================================================================
; Method Geov_Multifiles::get_levs
;
;======================================================================
Function geov_multifiles::get_levs
  return, (*self.files)[0]->get_levs()
End
;======================================================================
;
;======================================================================
FUNCTION geov_multifiles::get_interface_levs
  return, (*self.files)[0]->get_interface_levs()
End

;======================================================================
; Method Geov_Multifiles::get_levs2
;
;======================================================================
Function geov_multifiles::get_levs2
  return, (*self.files)[0]->get_levs2()
End

;======================================================================
; Method Geov_Multifiles::get_interface_levs2
;
;======================================================================
Function geov_multifiles::get_interface_levs2
  return, (*self.files)[0]->get_interface_levs2()
End

;======================================================================
; Method Geov_Multifiles::get_ilon
;
;======================================================================
Function geov_multifiles::get_ilon
  return, (*self.files)[0]->get_ilon()
End

;======================================================================
; Method Geov_Multifiles::get_ilat
;
;======================================================================
Function geov_multifiles::get_ilat
  return, (*self.files)[0]->get_ilat()
End

;======================================================================
; Method Geov_Multifiles::get_ilev
;
;======================================================================
Function geov_multifiles::get_ilev
  return, (*self.files)[0]->get_ilev()
End

;======================================================================
; Method Geov_Multifiles::get_ilevi
;
;======================================================================
Function geov_multifiles::get_ilevi
  return, (*self.files)[0]->get_ilevi()
End

;======================================================================
; Method Geov_Multifiles::get_itim
;
;======================================================================
Function geov_multifiles::get_itim
  return, (*self.files)[0]->get_itim()
End

;======================================================================
; Method Geov_Multifiles::get_pvars
;
;======================================================================
Function geov_multifiles::get_pvars
  return, (*self.files)[0]->get_pvars()
End

;======================================================================
; Method Geov_Multifiles::get_pvbits
;
;======================================================================
Function geov_multifiles::get_pvbits
  return, (*self.files)[0]->get_pvbits()
End

;======================================================================
; Method Geov_Multifiles::get_clabel
;
;======================================================================
Function geov_multifiles::get_clabel, ic
  return, (*self.files)[0]->get_clabel(ic)
End

;======================================================================
; Method Geov_Multifiles::get_cbit
;
;======================================================================
Function geov_multifiles::get_cbit, ic
  return, (*self.files)[0]->get_cbit(ic)
End

;======================================================================
; Method Geov_Multifiles::get_timestamp
;
;======================================================================
Function geov_multifiles::get_timestamp, time

  ; need the correct file for the time value
  file = self->get_file_obj(time)
  stamp=''
  if( obj_valid(file)) then stamp = file->get_timestamp( time )
  return, stamp
End

;======================================================================
; Method Geov_Multifiles::get_timestamps
;
;======================================================================
Function geov_multifiles::get_timestamps

  stamps = (*self.files)[0]->get_timestamps()

  for i = 1,self.nfiles-1 do stamps = [stamps,(*self.files)[i]->get_timestamps()]

  return, stamps

End

;======================================================================
; Method Geov_Multifiles::get_timetickvals
;
;======================================================================
PRO geov_multifiles::get_timetickvals, tickvals, ticklabels, times=times, minor=minor

  if (n_elements(times) gt 0) then ts = times else ts = *self.alltimes
  (*self.files)[0]->get_timetickvals, tickvals, ticklabels, times=ts, minor=minor

;  for i=1,self.nfiles-1 do begin
;    (*self.files)[i]->get_timetickvals, vals, labels
;    tickvals = [tickvals,vals]
;    ticklabels = [ticklabels,labels]
;  endfor

End

;======================================================================
; Method Geov_Multifiles::get_times
;
; Returns the time values of all time steps contained in the current
; set of netCDF files
;======================================================================
Function geov_multifiles::get_times
  return, self.alltimes
End

;======================================================================
; Method Geov_Multifiles::init
;
; Returns 1 if the criteria listed below are matched. Otherwise 0 is
; returned and the object will not be constructed successfully.
;
; Parameters:
;   filenames: An array containing the names of the files to be read
;              within one control panel.  All the files must meet the
;              following criteria:
;              * each file must be a legitimate geov_ncfile
;              * number of variables in the files must be equal
;              * names of variables in each of the files must be the same
;              * units of variables in each of the files must be the same
;              * number of dimensions in all of the files must be equal
;              * number of coordinates in all of the files must be equal
;              * time steps must be consecutive across all the files
;                with no time gaps or overlaps
;
;======================================================================
Function geov_multifiles::init, filenames

  nfiles = n_elements(filenames) ; number of files to check

  ; sort the filenames
  filenames = filenames[sort(filenames)]

  if (nfiles lt 1) then return, 0

  ; create array of goev_ncfiles
  files = objarr( nfiles )
  for ifile = 0, nfiles-1 do begin
    files[ifile] = obj_new( 'geov_ncfile',  filenames[ifile] )
    if ( not obj_valid( files[ifile] ) ) then return, 0
  endfor

  ; make sure the variables, dims, etc. match
  nvars0  = files[0]->get_nvars()
  vnames0 = files[0]->get_vnames()
  vndims0 = files[0]->get_vndims()
  vunits0 = files[0]->get_vunits()
  ndims0  = files[0]->get_ndims()
  ncoords0= files[0]->get_ncoords()

  for ifile = 1, nfiles-1 do begin
    nvars  = files[ifile]->get_nvars()
    ndims  = files[ifile]->get_ndims()
    ncoords= files[ifile]->get_ncoords()

    if ( nvars ne nvars0 ) then begin
      message, 'Number of varibles do not match'
      return, 0
    endif
    if ( ndims ne ndims0 ) then begin
      message, 'Number of dimensions do not match'
      return, 0
    endif
    if ( ncoords ne ncoords0 ) then begin
      message, 'Number of coordinates do not match'
      return, 0
    endif

    vnames = files[ifile]->get_vnames()
    vndims = files[ifile]->get_vndims()
    vunits = files[ifile]->get_vunits()

    for ivar=0, nvars0-1 do begin
      if ( ((*vnames0)[ivar]) ne ((*vnames)[ivar]) ) then begin
        message, 'Variable names do not match'
        return, 0
      endif
      if ( ((*vndims0)[ivar]) ne ((*vndims)[ivar]) ) then begin
        message, 'Variable dimensions do not match'
        return, 0
      endif
      if ( ((*vunits0)[ivar]) ne ((*vunits)[ivar]) ) then begin
        message, 'Variable units do not match'
        return, 0
      endif
    endfor
  endfor

  ; make sure there are no gaps in time or overlaps
  times = ptrarr(nfiles)
  ntimes= lonarr(nfiles)

  ; get time value arrays for each file
  for ifile = 0, nfiles-1 do begin
    times[ifile] = files[ifile]->get_times()
  endfor

  t0 = !VALUES.D_NAN
  delta0 = !VALUES.D_NAN

  if ptr_valid( times[0] ) then begin
   for ifile = 0, nfiles-1 do begin
    nts = n_elements( *times[ifile] )
    filetimes = *(times[ifile])

    if (ifile eq 0) then alltimes = filetimes $
    else alltimes = [alltimes, filetimes]

    for itime = 0, nts-1 do begin
      time = filetimes[itime]

      if( finite(t0) ) then begin
        delta = time-t0
        if (delta le 0) then begin
          message, 'Times overlap'
          return, 0
        endif

; do time steps really need to be consecutive ??
 ;       if ( finite(delta0) ) then begin
 ;         if (delta ne delta0) then begin
 ;           message, 'Time steps are non-consecutive'
 ;           return, 0
 ;         endif
 ;       endif
;
        delta0 = delta
      endif

      t0 = time
    endfor

    ntimes[ifile] = nts
   endfor
  endif

  ; initialize data members
  self.files  = ptr_new( files )
  self.nfiles = nfiles
  self.ntimes = ptr_new(ntimes )
  self.times  = ptr_new( times )
  self.alltimes  = ptr_new( alltimes )

  ; the seletion of files is valid -> return 'true'
  return, 1

End


;;======================================================================
;; Method Geov_Multifiles::get_timestamp
;;
;; Returns the index which cooresponds to the file containing the
;; specified time step index.
;;======================================================================
;Function geov_multifiles::get_file_index, time_index
;
;  it = 0
;
;  for ifile = 0, self.nfiles-1 do begin
;    ntimes = n_elements( *((*self.times)[ifile]) )
;    for itime = 0, ntimes-1 do begin
;      if ( it eq time_index ) then return, ifile
;      it = it+1
;    endfor
;  endfor
;
;  return, -1 ; index not found
;End

;======================================================================
; Method Geov_Multifiles::get_file_obj
;
; Returns the the file Geov_Ncfile object containing the
; specified time value.
;======================================================================
Function geov_multifiles::get_file_obj, time

  for ifile = 0, self.nfiles-1 do begin
    ntimes = n_elements( *((*self.times)[ifile]) )
    for itime = 0, ntimes-1 do begin
      t = ( *((*self.times)[ifile]) )[itime]
      file = (*self.files)[ifile]
      if ( t eq time ) then begin
        return, file
      endif
    endfor
  endfor

  return, ptr_new() ; ref not found
End

;======================================================================
; Method Geov_Multifiles::get_data
;
; Returns an array containing the data requested which is read from
; the current set of files.
;
; Parameters
;   vid:  id of the variable data to be read from the file set
;   offsets:  an array containing the offsets of the desired data set
;   counts: an array containing the counts of the desired data set
;======================================================================
FUNCTION geov_multifiles::get_data, vid, offsets, counts

                                ; we need to check for time dependence of the variable
ndims = (*(self->get_vndims()))[vid]
lastdid = self->get_dimid_of_var( ndims-1, vid)
d2coord = self->get_d2coord()
ic = (*d2coord)[lastdid]
itim = self->get_itim()

if( ic ne itim ) then begin     ; no time dependence...
    file = (*self.files)[0]
    data=file->get_data(vid, offsets, counts)
    result = ptr_new(*data)
endif else begin                ; the variable is time dependent

                                ; assume that the last dimension is the time dimension
    itd = ndims-1
    toffset = offsets[ itd ]
    tcount = counts[ itd ]

    myoffsets = offsets
    mycounts  = counts


                                ; we need to loop over the time indices and files to extract
                                ; data from multiple files and construct one array which contains
                                ; all the extracted data.

    offset= toffset
    ifile = 0
    finished = 0B
    totalcount = 0
    firstfile = 1B

    totalntimes = 0
    time_cnt = 0
    extracted_cnt = 0

    repeat begin

        nts = (*self.ntimes)[ifile]
        totalntimes = totalntimes + nts

        if ( totalntimes gt offset ) then begin

            fileoffset = offset

            if (firstfile) then begin
                fileoffset = offset-(totalntimes-nts)
                offset = 0
                firstfile = 0B
            endif

            file = (*self.files)[ifile]
            filetimes = (*self.ntimes)[ifile]
            filecnt = filetimes-fileoffset

            filecount = min( [tcount - extracted_cnt, filecnt] )
            totalcount = totalcount + filecount

            if ( totalcount ge tcount ) then finished=1B

            mycounts[itd] = filecount
            myoffsets[itd] = fileoffset

            data=file->get_data(vid, myoffsets, mycounts)
            extracted_cnt = extracted_cnt + mycounts[itd]

            if ( (size(*data))[0] eq 0 ) then *data = [*data] ; make array from scalor

            *data = reform( *data, mycounts, /overwrite )

            if(ptr_valid(result)) then begin
                                ; transpose the array so that they will 
                                ; be concatenated correctly

                revdims = REVERSE(mycounts)

                dat_trans = reform( transpose(*data) , revdims, /overwrite )

                res_trans = reform( transpose(*result) , res_revdims, /overwrite )

                res_trans = [res_trans,dat_trans]
                *result = transpose(res_trans) 

                time_cnt = time_cnt + revdims[0]

            endif else begin
                result = ptr_new(*data) ; this is the first file data is extracted from, so
                                ; let "result" be the array pointed to by "data"

                res_revdims = REVERSE(mycounts)
                time_cnt = res_revdims[0]
                
            endelse

            res_revdims[0] = time_cnt 

        endif

        ifile=ifile+1

    endrep until ((ifile eq self.nfiles) or (finished))

endelse                         ; the variable is time dependent

*result = reform( *result, /overwrite )
return, result

End

;======================================================================
; Method Geov_Multifiles::cleanup
;
;======================================================================
Function geov_multifiles::cleanup

  ; free the heap varaibles to prevent memory leakage

    for i=0, self.nfiles do begin
      obj_destroy, self.files[i]
    endfor

    ptr_free, self.files
    ptr_free, self.times
    ptr_free, self.ntimes
    ptr_free, self.alltimes
End

;======================================================================
; Object class representing a collection of netCDF files to be read by
; one control panel.
;
;======================================================================
Pro geov_multifiles__define
    struct = { geov_multifiles,    $ ; class name

               files:  ptr_new(),    $ ; ptr to array of objs of type geov_file
               nfiles: 0L,           $ ; number of files
               ntimes: ptr_new(),    $ ; ptr to array or number of time steps in each file
               times:  ptr_new(),    $ ; ptr to arrays of time values for each file
               alltimes: ptr_new()   $ ; ptr to array of all time values
              }
End
