function nctime::mk_timelabels, years, months, days, hours, minutes, seconds, format=format

  myyrs = years
                                ; year zero is not defined
  izeros = where(myyrs eq 0,nzs)
  if ( nzs gt 0 ) then begin
    myyrs(izeros) = -1
  endif

  juldays = julday(months,days,myyrs,hours,minutes,seconds)

  dummy = label_date(date_format=format)

  ;; format output string
  labels = StrArr(n_elements(juldays))
  for i=0L, n_elements(juldays)-1 do begin
    labels[i] = label_date(0,0,juldays[i])
  endfor

  return, labels

end

pro nctime::get_timetickvals, ticks, labels, times=times, minor=minor

  if n_elements(times) gt 0 then ts = times else ts = *self.times

  timeticks, ts, self.calendar, ticks, date_format, minor=minor

  if strcmp(self.calendar,'gregorian',4,/fold_case) then begin
    caldat, ticks, mons, days, yrs, hrs, mins, secs 
  endif else begin
    caldat_365, ticks, mons, days, yrs, hrs, mins, secs 
  endelse
 
  labels = self->mk_timelabels( yrs, mons, days, hrs, mins, secs, format=date_format )

end

function nctime::get_timelabels
  return, *self.timelabels
end

function nctime::get_timestamp, time

  times = *self.times
  labels = *self.timelabels

  for i=0,n_elements(times)-1 do begin
    if times[i] eq time then return, labels[i]
  endfor
  
  return, ''
end

pro nctime::get_times_and_labels, dates, seconds, times, labels, format=format

  years = dates/10000
  months = (dates - years*10000)/100
  days = dates - years*10000 - months*100
  hours = floor( seconds/3600. )
  minutes = (seconds - hours*3600.)/60.
  seconds = seconds - hours*3600. - minutes*60.

  ; year zero is not defined
;  izeros = where(years eq 0,nzs)
;  while nzs ne 0 do begin
;    years = years+1
;    izeros = where(years eq 0,nzs)
;  endwhile
 
  if strcmp(self.calendar,'gregorian',4,/fold_case) then begin
    ; year zero is not defined
    izeros = where(years eq 0,nzs)
    if ( nzs gt 0 ) then begin
      years(izeros) = -1
    endif
    times = julday(months,days,years,hours,minutes,seconds)
  endif else begin
    times = julday_365(months,days,years,hours,minutes,seconds) 
  endelse

  labels = self->mk_timelabels( years, months, days, hours, minutes, seconds, format=format)

end

function nctime::get_times
  return, *self.times
end

function nctime::get_timesptr
  return, self.times
end

pro nctime::cleanup
  if ptr_valid(self.times) then ptr_free, self.times
  if ptr_valid(self.timelabels) then ptr_free, self.timelabels
end

function nctime::init, fid, calendar=calendar, date_format=date_format

  tnames = ['date','datesec','time','month','t']

  fres = ncdf_inquire(fid)

  ntimes = -1

  ; deterimine which dimension is time dimension
  ; and get the time dimension size

  tdim = fres.recdim

  if (tdim gt -1 ) then begin
    ncdf_diminq, fid, tdim, tname, ntimes
  endif else begin
    for did =0,fres.ndims-1 do begin
      ncdf_diminq, fid, did, dname, size
      xx = where( strcmp( dname, tnames, /fold_case )  eq 1 , n )
      if (n gt 0) then begin
        tdim = did
        ntimes = size
        tname = tnames[xx]
        break
      endif
    endfor
  endelse

  ; abort if the time dimension can't be determined
  if (tdim eq -1) then return, 0


  date_vid = -1
  sec_vid = -1
  time_vid = -1
  alt_vid = -1 ; alternative time variable id
  found_cal = 0B

  
  ; loop through all variables to determine:
  ;  - calendar
  ;  - avaliable time type vars

  for vid =0,fres.nvars-1 do begin
    vres = ncdf_varinq(fid,vid)
    if ( strcmp( vres.name, 'date', /fold_case ) ) then begin
      date_vid = vid
      ;tdim = vres.dim[0]
      ;found_tim = 1B
    endif else if ( strcmp( vres.name, 'datesec', /fold_case ) ) then begin
      sec_vid = vid
    endif else if ( strcmp ( vres.name, 'time', /fold_case ) ) then begin
      time_vid = vid
      ;found_tim = 1B 
      ;tdim = vres.dim[0]
      natts = vres.natts
      for i = 0,natts-1 do begin
        attname = ncdf_attname( fid, time_vid, i )
        if ( strcmp ( attname, 'units', /fold_case ) ) then begin
          ncdf_attget, fid, vid, attname, attval
          units = string(attval)
          ;; analyse units
          theUnits = StrLowCase(StrCompress(StrTrim(units,2)))

          x =  strsplit(theUnits,' ', /extract )

          if (n_elements(x) ge 3 ) then begin
            if ( (x[0] eq 'days' or x[0] eq 'day') and ( x[1] eq 'since' ) ) then begin

                y = strsplit(x[2],'-',/extract)
                yr0  = fix(y[0])
                mon0 = fix(y[1])
                day0 = fix(y[2])
                if (n_elements(x) ge 4) then begin  
                  z = strsplit(x[3],':',/extract)
                  if (n_elements(z) ge 1) then hr0  = fix(z[0]) else hr0=0
                  if (n_elements(z) ge 2) then min0 = fix(z[1]) else min0=0
                  if (n_elements(z) ge 3) then sec0 = fix(z[2]) else sec0=0
                endif
            endif
          endif else begin
            Message, 'Units must be "days since YYYY-MM-DD [HH:MM:SS]"', /CONTINUE
          endelse

        endif
        if ( strcmp ( attname, 'calendar', /fold_case ) ) then begin
          ncdf_attget, fid, vid, attname, attval
          self.calendar = string(attval)
          found_cal = 1B
        endif
      endfor
    endif else if ( strcmp ( vres.name, tname, /fold_case ) ) then begin
      alt_vid = vid
    endif
  endfor

  if ( not found_cal  ) then begin
    if (n_elements(calendar) gt 0) then begin
      self.calendar = calendar
    endif 
  endif

  if ( date_vid gt -1 ) then begin
    result = NCDF_VARINQ(fid, date_vid)
    if (result.datatype eq "CHAR" ) then begin
      ncdf_varget, fid, date_vid, timevals
      self.timelabels = ptr_new(string(timevals))
    endif else begin

      ; use date var
      ncdf_varget, fid, date_vid, dates
      if (sec_vid gt -1) then begin
        ncdf_varget, fid, sec_vid, seconds
      endif else begin
        seconds = fltarr(ntimes)
        seconds[*] = 0.
      endelse

      self->get_times_and_labels, dates, seconds, times, labels, format=date_format

      self.timelabels = ptr_new(labels)
      self.times = ptr_new(times)
    endelse
  endif else if (time_vid gt -1) then begin

    result = NCDF_VARINQ(fid, time_vid)
    if (result.datatype eq "CHAR" ) then begin
      ncdf_varget, fid, time_vid, timevals
      self.timelabels = ptr_new(string(timevals))
    endif else begin

      ; use time var
      ncdf_varget, fid, time_vid, timevals
      ; get ref time

      if (n_elements(day0) gt 0) then begin
        if strcmp(self.calendar,'gregorian',4,/fold_case) then begin
          reftime = julday( mon0, day0, yr0, hr0, min0, sec0 )
        endif else begin
          reftime = julday_365( mon0, day0, yr0, hr0, min0, sec0 )
        endelse
      endif else begin
        reftime = 0
      endelse

      times = timevals + reftime
      self.times = ptr_new(times)

      if strcmp(self.calendar,'gregorian',4,/fold_case) then begin
        caldat, times, mons, days, yrs, hrs, mins, secs 
      endif else begin
        caldat_365, times, mons, days, yrs, hrs, mins, secs
      endelse

      labels = self->mk_timelabels( yrs,mons,days,hrs,mins,secs , format=date_format)
      self.timelabels = ptr_new(labels)
    endelse
  endif else if (alt_vid gt -1) then begin
    ncdf_varget, fid, alt_vid, timevals

    result = NCDF_VARINQ(fid, alt_vid)
    if (result.datatype eq "CHAR" ) then begin
      self.timelabels = ptr_new(string(timevals))
    endif else begin
      self.times = ptr_new(timevals)
    endelse
    
  endif else begin
    self.times = ptr_new(findgen(ntimes))
    self.timelabels = ptr_new( strtrim(string(indgen(ntimes)),2) )
  endelse

  if not ptr_valid( self.times ) then $
    self.times = ptr_new(findgen(ntimes))
  if not ptr_valid( self.timelabels ) then $
    self.timelabels = ptr_new( strtrim(string(*self.times),2) )

  return,1
end

pro nctime__define
  struct = { nctime,          $
             calendar:'gregorian', $
             reftime:0.d     , $
             times:ptr_new() , $ ; number of fractional days since refday
             timelabels:ptr_new()   $ 
           }
end

pro test_nctime
 help,/heap

  filenames = ['/ur/fvitt/data/ncdata/AAh0001.nc' $
              , '/ur/fvitt/data/ncdata/tmp.nc' $
              , '/ur/fvitt/data/ncdata/AAh0001.greg.nc' $
              , '/ur/fvitt/data/ncdata/AAh0001.nocal.nc' $
              , '/ur/fvitt/data/ncdata/fields_ncep_0103.nc' $
              , '/ur/fvitt/data/ncdata/climatology_monthly.nox.nc' $
              , '/ur/fvitt/data/ncdata/co4.nc' $
              , '/ur/fvitt/data/ncdata/emissions.c2h4.surface.T42LR.nc' $
              , '/ur/fvitt/data/ncdata/O3.nc' $
              , '/ur/fvitt/data/ncdata/wacam2.cam2.h0.0000-01.nc' ]

  filenames = file_search('/ur/fvitt/data/ncdata/*.nc')
;  filenames ='/ur/fvitt/data/ncdata/emissions.aircraft.T42LR.nc'
  nfiles = n_elements(filenames)

  for j=0,nfiles-1 do begin

    print, j,nfiles, ' ',filenames[j]
    print,'-------------------------'

    fid = ncdf_open( filenames[j], /nowrite )
    time = obj_new( 'nctime', fid )
    if obj_valid(time) then begin
      tlabels = time->get_timelabels()
      for i=0,n_elements(tlabels)-1 do print,tlabels[i]
      print,'-------------------------'

      if n_elements(tlabels) gt 1 then begin
        time->get_timetickvals, ticktimes, ticklabels
        for i=0,n_elements(ticktimes)-1 do print,ticktimes[i], '   ',ticklabels[i]
        print,'-------------------------'
      endif
      obj_destroy, time
    endif
    ncdf_close, fid

  endfor
 
  help,/heap

end
