;+
; NAME:
;        CHKSTRU  (function)
;
; PURPOSE:
;        Check validity of a structure and test if necessary
;        fields are contained. The result of this function will 
;        be 0 if the argument is not a structure (this includes
;        undefined values) or if the argument does not contain
;        all the tags requested with the FIELDS argument.
;
; CATEGORY:
;        General Programming
;
; CALLING SEQUENCE:
;        res=CHKSTRU(STRUCTURE,FIELDS [,/VERBOSE]) 
;
; INPUTS:
;        STRUCTURE --> the structure to be tested. If STRUCTURE is
;             not of type structure, the function will return 0
;
;        FIELDS --> a string or string array with field names to 
;             be contained in STRUCTURE. CHKSTRU returns 1 (true)
;             only if all field names are contained in STRUCTURE.
;             The entries of FIELDS may be upper or lowercase.
;
; KEYWORD PARAMETERS:
;        INDEX --> a named variable that will contain the indices of
;             the required field names in the structure. They can then
;             be assessed through structure.(index(i)) . Index will
;             contain -1 for all fields entries that are not in the
;             structure.
;
;        /VERBOSE --> set this keyword to return an error message 
;             in case of an error.
;
; OUTPUTS:
;        CHKSTRU returns 1 if successful, otherwise 0.
;
; SUBROUTINES:
;
; REQUIREMENTS:
;
; NOTES:
;        This routine should still work with IDL 5.0 or even 4.0
;
; EXAMPLE:
;        test = { a:1, b:2, c:3 }
;        required = ['a','c']
;        if CHKSTRU(test,required) then print,'found a and c.'
;
; MODIFICATION HISTORY:
;        mgs, 02 Mar 1998: VERSION 1.00
;        mgs, 07 Apr 1998: - second parameter (FIELDS) now optional
;        mgs, 23 Aug 2000: - changed copyright to open source
;                          - changed array indices to [] and "emacsified"
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2000 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################


FUNCTION chkstru,structure,fields,index=index,verbose=verbose
 

   ;; default index
   index = -1
 
   ;; first check number of parameters (must be at least 1)
   IF (n_params() LT 1) THEN BEGIN
      IF(keyword_set(verbose)) THEN $
         print,'CHKSTRU: ** invalid number of parameters ! **'
      RETURN,0
   ENDIF
 
 
   ;; check if the user really passed a structure
 
   s = size(structure)
   IF (s[1+s[0]] NE 8) THEN BEGIN
      IF(keyword_set(verbose)) THEN $
         print,'CHKSTRU: ** No structure passed ! **'
      RETURN,0
   ENDIF
 
   ;; only one parameter: then we are finished
   IF (n_params() EQ 1) THEN RETURN,1


 
   ;; see if required field names are contained in the structure
   ;; and return indices of these fields
 
   names = tag_names(structure)
   index = intarr(n_elements(fields)) - 1 ;; default index to 'not found'

   FOR i=0,n_elements(fields)-1 DO BEGIN
      ind = where(names EQ strupcase(fields[i]))
      IF (ind[0] LT 0) THEN BEGIN
         IF(keyword_set(verbose)) THEN $
            print,'CHKSTRU: ** Cannot find field '+fields[i]+' ! **'  
      ENDIF ELSE index[i] = ind[0]
   ENDFOR
 
 
   ;; check minimum value of index field: -1 indicates error
   RETURN,(min(index) GE 0)
 
END
 
