;+
; NAME:
;   MGS_LonVector (object)
;
; PURPOSE:
;     This object defines the longitude component of a rectangular
;   geophysical grid. Practically all of it's functionality is
;   inherited from the generic MGS_GridVector object; this object
;   simply constrains a few parameters (e.g. automatic setting of the
;   limit vector), and handles the computation of grid box weights.
;     The gridvector object (and, hence, the lonvector object) is
;   particular useful if one is dealing with different grids which may
;   have different resolutions or a different offset. A gridvector
;   allows you to:
;    (1) easily create a grid vector by supplying a list of values or
;        a few key parameters, 
;    (2) to retrieve grid vector indices for a given coordinate range,
;    (3) to retrieve coordinate values for an index array,
;    (4) to determine the fractional coverage of grid boxes for
;        a given coordinate range.
;     The grid vector defines a list of "center" coordinates and a
;   list of "edge" coordinates. The program can compute one from the
;   other assuming some regularity in the grid. The grid need not be
;   global in scope, nor need the grid box center coordinates be
;   equidistant. 
;     MGS_LonVector allows for cyclical coordinates (i.e. longitudes
;   can "wrap" on a global grid) and it deals with such coordinates
;   transparently. 
;
; AUTHOR:
;
;   Dr. Martin Schultz
;   Max-Planck-Institut fuer Meteorologie
;   Bundesstr. 55, D-20146 Hamburg
;   email: martin.schultz@dkrz.de
;
; CATEGORY:
;   General object programming
;
; CALLING SEQUENCE:
;
; NOTES:
;
; EXAMPLE:
;    See example procedure at the end of this file
;
; MODIFICATION HISTORY:
;   mgs, 17 Aug 2001:
;          - derived from generic GridVector object
;
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################


; =====================================================================================
; Grid vector information routines:
; GetIndex: Return the index vector for a given coordinate range
; GetValue: Return the coordinate values for a given index
; (unchanged from MGS_GridVector)

; =====================================================================================
; Grid vector creation and manipulation routines:
; PrintStatus : report some information about current settings
; Shift       : translate grid vector and impose or adjust limit
; Reverse     : reverse the order of the grid vector 
; Create      : create a new grid vector


; -------------------------------------------------------------------------------------
; ComputeWeights: (private)
;   This method computes the fractions covered by each longitude
; element, i.e. the ratio of the angle covered by the element over
; 360. degrees.

FUNCTION MGS_LonVector::ComputeWeights

   IF self.nvalues EQ 0 THEN RETURN, 0.

   result = ( *self.edges - shift(*self.edges,1) )/360.d0
   result = result[1:*]

   ;; Safety check: edge coordinates should be monotonously
   ;; increasing. In case they are not, warn and stop
   IF Max(result) LT 0. THEN stop, self.name+': Edges not monotonous!'

   RETURN, result
END


; -------------------------------------------------------------------------------------
; Create:
;   This method overwrites the Create method of the generic gridvector
; object. It simply calls the inherited Create method with the default
; limit set to 0 .. 360.

FUNCTION MGS_LonVector::Create, values, $
                                limit=limit, $
                                _Extra=extra
                                ; inherited keywords from MGS_GridVector:
                                ;   edges (boolean)
                                ;   type (string)
                                ;   nvalues (integer)
                                ;   start (float)
                                ;   increment (float)
                                ;   adjust_values (boolean)
                                ;   allow_wrap (boolean)


   IF N_Elements(limit) EQ 0 THEN limit = [ 0., 360. ]
   retval = self->MGS_GridVector::Create(values, $
                                         limit=limit, $
                                         _Extra=extra)

   RETURN, retval

END


; =====================================================================================
; Standard object methods:
;     GetProperty : retrieve object values
;     SetProperty : set object values
;     Cleanup     : free object pointers and destroy
; (unchanged from MGS_GridVector)
;     Init        : initialize object

; -------------------------------------------------------------------------------------
; Init:
; This method initializes the object values for the longitude
; vector. If no limit range is provided, the limit is set to [0.,
; 360.]. The object's name is defaulted to 'Longitude'. 
;   Since a longitude coordinate is cyclic if it spans the entire
; globe, the allow_wrap flag is set per default. Use the no_wrap
; keyword to prevent cycling for non-global longitude vectors.
; In all other aspects, initialisation is performed by
; GridVector::Init. 

FUNCTION MGS_LonVector::Init,  $
                  values,                 $ ; The grid vector values (optional)
                  limit=limit,            $ ; the range limit of the grid vector
                  no_wrap=no_wrap,        $ ; Do not allow coordinate cycling
                  name=name,              $ ; the object's name (default 'Longitude')
                  _Extra=extra  ; For future additions
                                ; Inherited keywords:
                                ; from MGS_GridVector:
                                ; type      : The grid vector type ('regular', 
                                ;             'irregular', 'gaussian')
                                ; edges     : Values denote edge coordinates 
                                ;             rather than centers (boolean)
                                ; nvalues   : Number of center coordinates
                                ; start     : First center coordinate value
                                ; increment : Increment between consecutive 
                                ;             coordinates (regular vector)
                                ; adjust_values : cut vectors if out of range
                                ;  
                                ; from MGS_BaseObject:
                                ; uvalue    : a user-defined value
                                ; no_dialog : Don't display message dialogs
                                ; debug     : Put object in debugging state


   ;; Set default name
   IF N_Elements(name) EQ 0 THEN name = 'Longitude'

   ;; Set default limit range to cover the globe
   IF N_Elements(limit) EQ 0 THEN limit = [0., 360.]

   ;; Determine wrap setting
   allow_wrap = 1 - Keyword_Set(no_wrap)

   RETURN, self->MGS_GridVector::Init(values, name=name, $
                                      allow_wrap=allow_wrap, $
                                      _Extra=extra)
END


; -------------------------------------------------------------------------------------
; This is the object definition for the gridvector object.
; NOTE:
; (1) If allow_wrap is set to 1, limit is only used to specify the
; allowed extent of the grid (the wrapping period).

PRO MGS_LonVector__Define

   objectClass = { MGS_LonVector,  $    ; The object class

                   INHERITS MGS_GridVector  $
                 }
                   
END

