;+
; NAME:
;   mgs_viewer (object)
;
; PURPOSE:
;   [well, to be honest: theoretical layout]
;     This object stores (scientific) data and the information
;   relevant for plotting and (simple) arithmetic analysis.
;   It provides methods for extracting data and computing
;   basic statistics.
;     It inherits from MGS_BaseObject which provides basic
;   functionality for displaying error messages and storing a user
;   value, and from IDL_Container which is used to store individual
;   variables. 
;     The BaseVariable object does not contain information or methods
;   for treatment of missing values or limited data range. These
;   are added in the definition of MGS_Variable.
;   Data of REAL, DOUBLE, LONG, INT, or BYTE type is fully supported,
;   for other types BaseVariable serves as a container object only
;   (i.e. you cannot compute a mean, extract a range etc.).
;
; AUTHOR:
;
;   Dr. Martin Schultz
;   Max-Planck-Institut fuer Meteorologie
;   Bundesstr. 55, D-20146 Hamburg
;   email: martin.schultz@dkrz.de
;
; CATEGORY:
;   General Science, Objects
;
;
; CALLING SEQUENCE:
;
;
;
; EXAMPLE:
;
;
;
; MODIFICATION HISTORY:
;   Martin G. Schultz, 30 May 2000: VERSION 1.00
;
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2000 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################


; -------------------------------------------------------------------------------------

; =====================================================================================
; Dummy methods for testing:
;     Select : just set a specific variable name and call the LoadData
;              method
;     Show   : display a contour plot

pro MGS_Viewer::Select

   ; *** DUMMY : set variable name to TRA14
   thisVarname = 'TRA14'

   ;; retrieve the first data object
   theData = self.theData->Get(position=0)
   IF not Obj_Valid(theData) THEN RETURN

   ;; load the variable data into memory
   theData->Loaddata, thisVarname

end


; -------------------------------------------------------------------------------------

pro MGS_Viewer::Show

   ; *** DUMMY : always draw a contour plot 

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, !Error_State.Msg
      RETURN
   ENDIF
   
   ;; retrieve the first data object
   theData = self.theData->Get(position=0)
   IF not Obj_Valid(theData) THEN RETURN

   ;; retrieve the first variable from the data object
   theVar = theData->Get(position=0)
   theVar->GetProperty, name=name, units=units, data=data

help,name,units,data

   ndims = Size(data,/N_Dimensions)
   dims = Size(data, /Dimensions)
   ok = 0
   CASE ndims OF
      0 : print,'Scalar variable. Value=', data   
      1 : plot, data   ; 1-D
      2 : ok = 1
      3 : BEGIN
          data = data[*,*,dims[2]-1]
          ok = 1
          END
      4 : BEGIN
          data = data[*,*,dims[2]-1, 0]
          ok = 1
          END
      ELSE : print,'Higher dimensions not supported!'
   ENDCASE

   IF not ok THEN RETURN

   ;; create contour plot
   minval = min(data, max=maxval)
   print,'Minimum and maximum of data : ',minval,maxval

   scale = 1.e-9

   loadct, 27, bottom=2, ncolors=20
   c_col = indgen(20)+2
   level = findgen(20)*50

   contour, data/scale, level=level, c_color=c_col

end
   

; =====================================================================================
; Methods overwriting parent methods:
;     Validate    : make sure variable names are consistent with container

; -------------------------------------------------------------------------------------
; Validate: 
;   Make sure the variable names as stored in the varnames field are
; consistent with the names of the variables stored in the container.

pro Mgs_Viewer::Validate

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, !Error_State.Msg
      RETURN
   ENDIF

   ;; Call parent's Validate method first
   self->MGS_BaseObject::Validate

end


; =====================================================================================
; Standard object methods:
;     GetProperty : retrieve object values
;     SetProperty : set object values
;     Cleanup     : free object pointers and destroy
;     Init        : initialize object

; -------------------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user. Varnames returns information from the first variable of each
; data object.

pro Mgs_Viewer::GetProperty,  $
                   varnames=varnames,      $ ; The variable names of the data objects
                   filenames=filenames,    $ ; The filenames of the attached data files
                   _Ref_Extra=extra
                                ; Inherited keywords:
                                ; name      : The variable name
                                ; uvalue    : a user-defined value


   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, !Error_State.Msg
      RETURN
   ENDIF

   ;; Call GetProperty method from parent object
   self->MGS_BaseObject::GetProperty, _Extra=extra

   ;; get filenames of all data objects
   IF Arg_Present(filenames) THEN BEGIN
       allData = self.theData->Get(/All)
       filenames = StrArr(N_Elements(allData))
       FOR i=0L, N_Elements(allData)-1 DO BEGIN
           allData->GetProperty, filename=thisName
           filenames[i] = thisName
       ENDFOR
   ENDIF

   ;; get first variable names of all data objects
   IF Arg_Present(varnames) THEN BEGIN
       IF N_Elements(allData) EQ 0 THEN $
         allData = self.theData->Get(/All)
       varnames = StrArr(N_Elements(allData))
       FOR i=0L, N_Elements(allData)-1 DO BEGIN
           varnames[i] = allData->Get(0)
       ENDFOR
   ENDIF

end

; -------------------------------------------------------------------------------------
; SetProperty:
; ??? *** ####
;   This method sets specific object values. Here, only a filename can
; be set, which causes a previously opened netCDF file to be closed
; and the new file to be opened. If variables had already been loaded
; from the old file, the isclean flag is set to 0 to indicate that the
; variables are no longer linked to the file they were retrieved from. 

pro Mgs_Viewer::SetProperty,  $
                   _Ref_Extra=extra
                                ; Inherited keywords:
                                ; name      : The variable name
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, !Error_State.Msg
      RETURN
   ENDIF

   ;; Call SetProperty method of BaseObject
   self->MGS_BaseObject::SetProperty, no_copy=no_copy, _Extra=extra

end

; -------------------------------------------------------------------------------------
; Cleanup:
; This method frees all data stored in the object.

pro Mgs_Viewer::Cleanup

   ;; Destroy linked objects
   IF Obj_Valid(self.theData) THEN Obj_Destroy, self.theData

   ;; Call parents' cleanup method
   self->MGS_BaseObject::Cleanup

end

; -------------------------------------------------------------------------------------
; Init:
;   This method initializes the object values. 
; The first data object is created and will receive the optional
; filename keyword in which case it will be attached to a (netCDF)
; file. Not much else happens. Use the Select method to choose a
; variable for plotting.

function Mgs_Viewer::Init,  $
                   _Ref_Extra=extra
                                ; Inherited keywords (from MGS_BaseObject):
                                ; name      : The viewer name
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : A user-defined value
                                ; Inherited keywords (from MGS_Data):
                                ; filename  : The name of a netCDF file


   ;; Initialize parent objects first
   IF not self->MGS_BaseObject::Init(_Extra=extra) THEN RETURN, 0

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error initializing object!'
      RETURN, 0
   ENDIF

   ;; Initialize data object
   theData = Obj_New('MGS_Data', _Extra=extra)
   IF not Obj_Valid(theData) THEN BEGIN
       self->ErrorMessage, 'Cannot initialize data object!'
       RETURN, 0
   ENDIF

   ;; Initialize container
   self.theData = Obj_New('IDL_Container')
   IF not Obj_Valid(self.theData) THEN BEGIN
       self->ErrorMessage, 'Cannot initialize data object!'
       RETURN, 0
   ENDIF

   ;; Add data object to container
   self.theData->Add,theData

   ;; Reset error state variable
   Message, /Reset

   return, 1
end




; -------------------------------------------------------------------------------------
; This is the object definition. Derived objects should create a new
; structure and append these fields via the INHERITS MGS_BaseObject
; syntax.

pro Mgs_Viewer__Define

   objectClass = { Mgs_Viewer,  $    ; The object class
                   theData  : Obj_New('IDL_Container'),  $  ; Container for data objects
                   INHERITS MGS_BaseObject  $
                 }
                   

end
