;-------------------------------------------------------------
;+
; NAME:
;    mgsWBox
;
; PURPOSE:
;    Superclass for pages, visual areas, frames and graphics
;    windows.
;
; CATEGORY:
;    Graphics, objects
;
; PROPERTIES:
;    public properties:
;       X0, X1 : lower left corner of box
;       XWidth, YWidth : lenght and height of box
;       Unit : measuring unit of the above [pixel,cm,in[ch]]
;       Title : a title string (used only in subclasses)
;    private properties:
;       DX0, DY0 : lower left corner of box in private unit (pixel)
;       DX1, DY1 : upper right corner of box in private unit
;       DXWidth, DYWidth : size of box in private unit 
;
; METHODS:
;    public methods:
;       GetProperty
;       SetProperty
;       Show
;       Init
;       Cleanup
;    private methods:
;       ComputePrivateCoordinates
;       ChangeUnit
;       CheckUnit
;    
; DESCRIPTION:
;    WBox defines a rectangular area and handles unit conversion as 
;    well as size changes of this box. It is meant as parent object for
;    Page, V(iewport)Area, Frame, and GrWindow.
;
; INTERFACES:
;
; SIDE EFFECTS:
;
; EXAMPLE:
;
; MODIFICATION HISTORY:
;    12 Nov 1999 :  written by Martin G. Schultz
;-
; Copyright
;-------------------------------------------------------------


; **********************************************************************
; **********************************************************************
; Get Properties of mgsWBox
;
pro mgsWBox::GetProperty, X0=X0,           $
                          Y0=Y0,           $
                          XWidth=XWidth,   $
                          YWidth=YWidth,   $
                          Unit=Unit,       $
                          Title=Title,     $
                          _ref_extra=extra

   if arg_present(X0) then X0 = self.X0
   if arg_present(Y0) then Y0 = self.Y0
   if arg_present(XWidth) then XWidth = self.XWidth
   if arg_present(YWidth) then YWidth = self.YWidth
   if arg_present(Unit) then Unit = self.Unit
   if arg_present(Title) then Title = self.Title

end  ;  mgsWBox::GetProperty


; **********************************************************************
; **********************************************************************
; Set Properties of mgsWBox
;
pro mgsWBox::SetProperty, X0=X0,           $
                          Y0=Y0,           $
                          XWidth=XWidth,   $
                          YWidth=YWidth,   $
                          Unit=Unit,       $
                          Title=Title,     $
                          Error=Error,     $
                          _ref_extra=extra


   ; A permanent unit change is only performed if Unit is
   ; the only keyword used. Otherwise it is assumed that 
   ; only the altered values shall be be interpreted in 
   ; the given unit.
   ; If no unit is given, the arguments are assumed to be
   ; in the current unit.
; print,'##   In mgsWBox::SetProperty'
   Error = 0

   ; save current unit
   thisUnit = self.Unit

   ; check validity of a given unit
   if n_elements(Unit) eq 1 then begin
      ok = self->CheckUnit(Unit,newUnit)
      if not ok then begin
         ok2 = Error_Message('Invalid Unit!')
         Error = 1
         return
      endif
      ; convert dimensions to new unit
      ok = self->ChangeUnit(thisUnit,newUnit)
   endif

   ; set new dimensions
   if n_elements(X0) eq 1 then self.X0 = double(X0)
   if n_elements(Y0) eq 1 then self.Y0 = double(Y0)
   if n_elements(XWidth) eq 1 then self.XWidth = abs(double(XWidth))
   if n_elements(YWidth) eq 1 then self.YWidth = abs(double(YWidth))

   ; convert unit back to old unit if dimensions were changed 
   if (n_elements(Unit) eq 1) AND $
                 (n_elements(X0) +      $
                  n_elements(Y0) +      $
                  n_elements(XWidth) +  $
                  n_elements(YWidth) gt 0) then begin
      ok = self->ChangeUnit(newUnit,thisUnit)
   endif

   ; update values of standard coordinates
   if not self->ComputePrivateCoordinates() then begin
      ok2 = Error_Message('Erroneous size parameters!')
      Error = 2
      return
   endif

   if n_elements(Title) eq 1 then self.Title = StrTrim(Title,2)

end  ;  mgsWBox::SetProperty


; **********************************************************************
; **********************************************************************
; Check if unit is defined
;
function mgsWBox::CheckUnit, Unit, thisUnit

   ; thisUnit returns unit in lowercase and without blanks
   ; use thisUnit only if CheckUnit returns 1
   thisUnit = StrTrim(StrLowCase(Unit),2)

   if thisUnit eq 'inch' then thisUnit = 'in'

   return, ( thisUnit eq 'cm' OR      $
             thisUnit eq 'in' OR      $
             thisUnit eq 'pixel')


end  ;  mgsWBox::CheckUnit


; **********************************************************************
; **********************************************************************
; Change Unit
;
function mgsWBox::ChangeUnit, FromUnit, ToUnit

   ; This method changes the unit entry and recomputes the
   ; public property values X0, Y0, XWidth and YWidth for
   ; the new unit. Here we assume pixels as standard unit.
   ; The unit names must be valid (check with self->CheckUnit)

   success = 1

   case fromUnit of
      'cm' : begin
               factor1x = !D.X_PX_CM
               factor1y = !D.Y_PX_CM
             end
      'in' : begin
               factor1x = !D.X_PX_CM * 2.54
               factor1y = !D.Y_PX_CM * 2.54
             end
      else : begin
               factor1x = 1.0
               factor1y = 1.0
             end
   end

   case toUnit of
      'cm' : begin
               factor2x = !D.X_PX_CM
               factor2y = !D.Y_PX_CM
             end
      'in' : begin
               factor2x = !D.X_PX_CM * 2.54
               factor2y = !D.Y_PX_CM * 2.54
             end
      else : begin
               factor2x = 1.0
               factor2y = 1.0
             end
   end

   self.X0 = self.X0 * factor1x / factor2x
   self.Y0 = self.Y0 * factor1y / factor2y
   self.XWidth = self.XWidth * factor1x / factor2x
   self.YWidth = self.YWidth * factor1y / factor2y

   self.Unit = toUnit

   return,success 

end  ;   mgsWBox::ChangeUnit


; **********************************************************************
; **********************************************************************
; Convert coordinates to standard system
;
function mgsWBox::ComputePrivateCoordinates

   ; in this SuperClass we assume standard unit is 'pixel'
   ; convert user values to standard, copy them, and convert back

   thisUnit = self.Unit
   ok = self->ChangeUnit(thisUnit,'pixel')
   if not ok then begin
      ok2 = Error_Message('Error in unit conversion!')
      return,0
   endif

   self.DX0 = self.X0
   self.DXWidth = self.XWidth
   self.DX1 = self.DX0 + self.DXWidth
   self.DY0 = self.Y0
   self.DYWidth = self.YWidth
   self.DY1 = self.DY0 + self.DYWidth

   ok = self->ChangeUnit('pixel',thisUnit)

   return,ok

end  ;  mgsWBox::ComputePrivateCoordinates


; **********************************************************************
; **********************************************************************
; Display mgsWBox
;
; This routine produces no output.
pro mgsWBox::Show

end  ;  mgsWBox::Show


; **********************************************************************
; **********************************************************************
; Cleanup method
;
pro mgsWBox::CleanUp

   print,'mgsWBox::CleanUp'

end  ;  mgsWBox::CleanUp


; **********************************************************************
; **********************************************************************
; Initialization method
;
function mgsWBox::Init,   $
                    X0=X0,              $
                    Y0=Y0,              $
                    XWidth=XWidth,      $
                    YWidth=YWidth,      $
                    Unit=Unit,          $
                    Title=Title,        $
                    _ref_extra=extra


   ; check inputs and set defaults 
   if n_elements(X0) eq 0 then X0 = 0.
   if n_elements(Y0) eq 0 then Y0 = 0.
   if n_elements(XWidth) eq 0 then XWidth = 0.
   if n_elements(YWidth) eq 0 then YWidth = 0.
   if n_elements(Unit) eq 0 then Unit = 'pixel'
   if n_elements(Title) eq 0 then Title = ''
   thisUnit = StrLowCase(Unit)

   ; Try to set properties
   ; Need to set Unit first to make it permanent
   ok = self->CheckUnit(thisUnit,newUnit)
   if not ok then begin
      ok2 = Error_Message('Invalid Unit!')
      return,0
   endif
   self.Unit = newUnit
   ; Now set all other properties
   self->mgsWBox::SetProperty,X0=X0,Y0=Y0,XWidth=XWidth,YWidth=YWidth, $
           Title=Title,Error=Error

print,'##',self.x0,self.y0,self.xwidth,self.ywidth,'>',self.unit,'<'

   return,(Error eq 0)

end  ;  mgsWBox::Init


; **********************************************************************
; **********************************************************************
; Structure definition
;
pro mgsWBox__Define

   temp = {mgsWBox,             $
                    X0 : 0.D0,          $
                    Y0 : 0.D0,          $
                    XWidth : 0.D0,      $
                    YWidth : 0.D0,      $
                    Unit : '',          $
                    Title : '',         $
                    DX0 : 0.D0,         $
                    DY0 : 0.D0,         $
                    DX1 : 0.D0,         $
                    DY1 : 0.D0,         $
                    DXWidth : 0.D0,     $
                    DYWidth : 0.D0 }

end  ;  mgsWBox__Define



